/**
 * TabulierendBaukasten.java
 * eu.gronos.kostenrechner.controller (Kostenrechner)
 */
package eu.gronos.kostenrechner.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.swing.JTable;

import eu.gronos.kostenrechner.data.tenordaten.BegruendungsElemente;
import eu.gronos.kostenrechner.data.tenordaten.BegruendungsZahlenZeile;
import eu.gronos.kostenrechner.data.tenordaten.CaretRange;
import eu.gronos.kostenrechner.data.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.interfaces.Tabulierend;
import eu.gronos.kostenrechner.interfaces.TenorVorbereitend;
import eu.gronos.kostenrechner.view.result.TenorDialog;

/**
 * Die Klasse speichert die BegründungsTabelle eines {@link Tabulierend} und
 * hält Methoden zum Umgang damit bereit: So erstellt
 * {@link #toTableCellValues()} die eigentlichen Tabellendaten für
 * {@link Tabulierend#getTableCellValues()} und {@link #toColumnHeaders()}
 * erstellt die Kopfdaten für die Tabelle.
 *
 * @author Peter Schuster (setrok)
 * @date 17.08.2018
 *
 */
public class BegruendungsZahlenTabelle {
	private final CaretRange range;
	private final List<BegruendungsZahlenZeile> zahlenZeilen;// DoubleDataRows
	private final List<String> columnHeaders;
	private final List<String> rowHeaders;
	private int[] columnWeight = null;
	private int tableWeight = -1;

	/**
	 * Der Standardkonstruktor legt alle Listen selbst an
	 * 
	 */
	public BegruendungsZahlenTabelle() {
		this.range = new CaretRange();
		this.zahlenZeilen = new ArrayList<BegruendungsZahlenZeile>();// DoubleDataRows
		this.columnHeaders = new ArrayList<String>();
		this.rowHeaders = new ArrayList<String>();
	}

	/**
	 * Der Konstruktor für den, der schon alles hat (sprich: alle Tabellendaten),
	 * braucht diese in drei Listen
	 * 
	 * @param doubleValues  {@link #zahlenZeilen}
	 * @param columnHeaders {@link #columnHeaders}
	 * @param rowHeaders    {@link #rowHeaders}
	 * @param range         {@link #range}
	 */// DoubleDataRows
	public BegruendungsZahlenTabelle(List<BegruendungsZahlenZeile> doubleValues, List<String> columnHeaders,
			List<String> rowHeaders, CaretRange range) {
		this.range = range;
		this.zahlenZeilen = new ArrayList<BegruendungsZahlenZeile>(doubleValues);// DoubleDataRows
		this.columnHeaders = new ArrayList<String>(columnHeaders);
		this.rowHeaders = new ArrayList<String>(rowHeaders);
	}

	/**
	 * Die Methode baut Tabellendaten aus {@link #rowHeaders} und
	 * {@link #zahlenZeilen}. Die Spaltenüberschriften gibt sie nicht zurück,
	 * sondern die Methode {@link #toColumnHeaders()}.
	 * 
	 * @return ein Array Object[][]
	 */
	public Object[][] toTableCellValues() {
		if ((zahlenZeilen == null || columnHeaders == null || rowHeaders == null)
				|| (zahlenZeilen.size() < 1 || columnHeaders.size() < 1 || rowHeaders.size() < 1)) {
			return null;
		}
		if (zahlenZeilen.size() != rowHeaders.size()) {
			throw new IllegalArgumentException("doubleValues und rowHeaders müssen gleich viele Zeilen haben!");
		}
		/*
		 * Die Tabelle hat so viele Zeilen wie tableCellValues groß ist und so viele
		 * Spalten wie columnHeaders groß ist. Denn: Erste Zeile sind NICHT die
		 * Spaltenüberschriften (die werden gesondert ausgegeben)
		 */
		Object[][] tableCellValues = new Object[zahlenZeilen.size()][columnHeaders.size()];
		/*
		 * Jede Zeile besteht aus einem Stück rowHeaders und einer Zeile
		 * tableCellValues. ACHTUNG: Dabei hat doubleValues eine Spalte weniger als
		 * columnHeaders!
		 */
		for (int row = 0; row < zahlenZeilen.size(); row++) {
			tableCellValues[row][0] = rowHeaders.get(row);
			for (int column = 1; column < columnHeaders.size(); column++) {
				BegruendungsZahlenZeile numberRow = zahlenZeilen.get(row);// DoubleDataRows
				if (numberRow != null && numberRow.size() > column - 1) {
					tableCellValues[row][column] = numberRow.get(column - 1);
				} else {
					System.out.printf("Null oder IndexOutOfBounds bei (%d, %d)!%n", row, column);
				}
			}
		}
		return tableCellValues;
	}

	/**
	 * Die Methode konvertiert die {@link ArrayList}&lt;{@link String}>
	 * {@link #columnHeaders} zu einem Array String[] mit den
	 * Kopfdaten/Spaltenüberschriften für {@link Tabulierend#getColumnHeaders()}.
	 * 
	 * @return ein Array String[] mit den Kopfdaten/Spaltenüberschriften
	 */
	public String[] toColumnHeaders() {
		if (columnHeaders == null || columnHeaders.isEmpty()) {
			return null;
		}
		String[] array = new String[columnHeaders.size()];
		return columnHeaders.toArray(array);
	}

	/**
	 * Die Methode gibt die {@link ArrayList}&lt;{@link String}> {@link #rowHeaders}
	 * als {@link Collections#unmodifiableList(List)} mit den
	 * Kopfdaten/Spaltenüberschriften für
	 * {@link TenorDatenContainer}/{@link BegruendungsElemente#columnHeaders}
	 * zurück.
	 * 
	 * columnHeaders enthält die Kopfdaten/Spaltenüberschriften als List&lt;String>.
	 * Diese werden hinterher zur ersten Zeile/Überschriftenzeile der Tabelle
	 * 
	 * @return eine {@link List}&lt;{@link String}> mit den
	 *         Kopfdaten/Spaltenüberschriften
	 */
	public List<String> getColumnHeaders() {
		if (columnHeaders == null || columnHeaders.isEmpty()) {
			return null;
		}
		return Collections.unmodifiableList(columnHeaders);
	}

	/**
	 * Die Methode gibt die {@link ArrayList}&lt;{@link String}> {@link #rowHeaders}
	 * als {@link Collections#unmodifiableList(List)} mit den Zeilenüberschriften
	 * für {@link TenorDatenContainer}/{@link BegruendungsElemente#rowHeaders}
	 * zurück.
	 * 
	 * rowHeaders enthält die Zeilenüberschriften als List&lt;String>. Diese werden
	 * zur ersten Spalte der Tabelle.
	 * 
	 * @return eine {@link List}&lt;{@link String}> mit den Zeilenüberschriften
	 */
	public List<String> getRowHeaders() {
		if (rowHeaders == null || rowHeaders.isEmpty()) {
			return null;
		}
		return Collections.unmodifiableList(rowHeaders);
	}

	/**
	 * Die Methode gibt die {@link ArrayList}&lt;{@link Double}>
	 * {@link #zahlenZeilen} als {@link Collections#unmodifiableList(List)} Double[]
	 * mit den Rohdaten für
	 * {@link TenorDatenContainer}/{@link BegruendungsElemente#zeilenZahlen}.
	 * 
	 * doubleValues enthält die eigentlichen Tabellendaten (die Double-Werte ohne
	 * Zeilen- oder Spalten-Überschriften) als List&lt;Double[]>.
	 * 
	 * @return eine {@link List}&lt;{@link Double}[]> mit den Rohdaten
	 */
	public List<BegruendungsZahlenZeile> getDoubleData() {// DoubleDataRows
		if (zahlenZeilen == null || zahlenZeilen.isEmpty()) {
			return null;
		}
		return zahlenZeilen;
	}

	/**
	 * Die Methode sagt, welche Datentypen von {@link #toTableCellValues()}
	 * zurückgegeben werden.
	 * 
	 * @return ein Array Class&lt;?>[], das so groß ist wie {@link #columnHeaders}
	 *         und deren erstes Objekt {@link String#getClass()} ist und alle
	 *         übrigen {@link Double#getClass()}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Tabulierend#getColumnClasses()
	 */
	public Class<?>[] toColumnClasses() {
		Class<?>[] columnClasses;
		if (columnHeaders == null || columnHeaders.size() < 1)
			return null;
		columnClasses = new Class<?>[columnHeaders.size()];
		columnClasses[0] = String.class;
		for (int i = 1; i < columnHeaders.size(); i++)
			columnClasses[i] = Number.class;//Double.class;
		return columnClasses;
	}

	/**
	 * Die Methode dient dazu, aus der {@link #zeilenListe} einen Text mit
	 * Tabulatoren zu basteln, der von der {@link JTable} im {@link TenorDialog}
	 * überlagert werden kann. Gleichzeitig wird auch
	 * {@link eu.gronos.kostenrechner.interfaces.Tabulierend#getRange() range}
	 * gesetzt.
	 * 
	 * @param gruende ein {@link StringBuilder} mit den Gründen, der auch verändert
	 *                wird
	 * @return gibt die veränderten gruende auch wieder zurück
	 */
	public StringBuilder toStringBuilder(StringBuilder gruende) {
		range.beginn = gruende.length() - 1;
		// Erst die Überschriften
		for (int index = 0; index < columnHeaders.size(); index++) {
			gruende.append(columnHeaders.get(index));
			// Tabulator oder Zeilenumbruch.
			if (index == columnHeaders.size() - 1) {
				gruende.append("\n");
			} else {
				gruende.append("\t");
			}
		}
		// dann die Double-Werte
		for (int index = 0; index < zahlenZeilen.size(); index++) {
			gruende.append(rowToText(index));
		}
		range.ende = gruende.length() - 1;
		return gruende;
	}

	/**
	 * Die Methode fügt der Tabelle die Spaltenüberschriften hinzu
	 * 
	 * @param values eine List&lt;String> mit Spaltenüberschriften. Muss eine Spalte
	 *               enthalten, als die Doublewerte haben.
	 * @return true, wenn es geklappt hat
	 */
	public boolean add(List<String> values) {
		return columnHeaders.addAll(values);
	}

	/**
	 * Die Methode fügt der Tabelle eine Zeile hinzu, bestehend aus einer
	 * Zeilenüberschrift und einem Array mit Double-Werten
	 * 
	 * @param header Zeilenüberschrift als String
	 * @param values Tabellenwerte als List&lt;Double>
	 * @return true, wenn es geklappt hat
	 * 
	 * @see List#add(Object)
	 */
	public boolean add(String header, BegruendungsZahlenZeile values) {// DoubleDataRows
		if (header == null || values == null) {
			throw new NullPointerException();
		}
		return rowHeaders.add(header) && zahlenZeilen.add(new BegruendungsZahlenZeile(values));// DoubleDataRows
	}

	/**
	 * range speichert Beginn und Ende der Begründungstabelle im Begründungstext als
	 * int[].
	 * 
	 * @return
	 * 
	 * @return gibt {@link #range} als int [] zurück.
	 * 
	 * @see Tabulierend#getRange()
	 */
	public CaretRange getRange() {
		return range;
	}

	/**
	 * Die Methode hilft {@link TenorVorbereitend}-Klassen, ein
	 * {@link TenorDatenContainer} zu befüllen. Dazu liefert es die nötigen Daten
	 * für ein {@link BegruendungsElemente} zurück.
	 * 
	 * @param gruende Die Gründe selbst kann {@link BegruendungsZahlenTabelle} nicht
	 *                erzeugen, deshalb muss die aufrufende Methode sie übergeben.
	 * @return das erstellte {@link BegruendungsElemente}
	 */
	public BegruendungsElemente toBegruendungsElemente(String gruende) {
		BegruendungsElemente begruendung = new BegruendungsElemente();
		begruendung.text = gruende;
		begruendung.range = getRange();
		begruendung.columnHeaders = getColumnHeaders();
		begruendung.rowHeaders = getRowHeaders();
		begruendung.zeilenZahlen = getDoubleData();
		return begruendung;
	}

	/**
	 * @return gibt {@link #columnWeight} für die mit <code>index</code> bezeichnete
	 *         Spalte als <code>int</code> zurück.
	 */
	public int getColumnWeight(int index) {
		if (columnWeight == null) {
			return -1;
		} else if (columnWeight.length - 1 < index) {
			return -1;
		} else if (index < 0) {
			return -1;
		} else
			return columnWeight[index];
	}

	/**
	 * @return gibt {@link #tableWeight} als {@link int} zurück.
	 */
	public int getTableWeight() {
		return tableWeight;
	}

	/**
	 * Die Methode errechnet das {@link #columnWeight} und das {@link #tableWeight},
	 * damit man es mit {@link #getColumnWeight(int)} und mit
	 * {@link #getTableWeight()} erfragen kann.
	 */
	public void calculateTableWeight() {
		columnWeight = new int[columnHeaders.size()];
		tableWeight = 0;
		// Erst die Überschriften
		for (int index = 0; index < columnHeaders.size(); index++) {
			gainWeight(index, columnHeaders.get(index).length());
		}
		// dann die Double-Werte
		for (int index = 0; index < zahlenZeilen.size(); index++) {
			gainWeight(0, rowHeaders.get(index).length());
			// List<Number>
			BegruendungsZahlenZeile row = zahlenZeilen.get(index);// Double
			for (int cell = 0; cell < row.size(); cell++) {
				final String cellText = cellText(row, cell);
				gainWeight(cell + 1, cellText.length());
			}
		}

	}

	/**
	 * Die Methode wandelt die über den index angegebene Tabellenzeile, bestehend
	 * aus einem String aus {@link #rowHeaders} und einer Zeile aus
	 * {@link #zahlenZeilen} in eine Tab-separierte Textzeile als {@link String} um
	 * 
	 * @param index die Zeile aus {@link #zahlenZeilen}
	 * @return eine Tab-separierte Textzeile (String)
	 */
	private String rowToText(int index) {
		if (zahlenZeilen == null || rowHeaders == null || zahlenZeilen.size() <= index || rowHeaders.size() <= index) {
			return "";
		}
		StringBuffer text = new StringBuffer();
		text.append(rowHeaders.get(index));
		text.append("\t");
		text.append(zahlenZeilen.get(index).tabSeparatedValues());
		return text.toString();
	}
//		List<Number>
		// Double
//		for (int cell = 0; cell < row.size(); cell++) {
//			final String cellText = cellText(row, cell);
//			text.append(cellText);
//			// Tabulator oder Zeilenumbruch.
//			if (cell == row.size() - 1) {
//				text.append("\n");
//			} else {
//				text.append("\t");
//			}
//		}

	/**
	 * Die Methode baut einen formattierten {@link String} zusammen.
	 * 
	 * @param row  Nummer der Zeile
	 * @param cell Nummer der Spalte
	 * @return den formattierten {@link String}
	 */
	private String cellText(BegruendungsZahlenZeile row, int cell) {// List<Number>Double
		return row.numberToSpalte(row.get(cell));
		// return String.format("%,.2f", row.get(cell));
	}

	/**
	 * Die Methode zählt dem {@link #columnWeight} für die mit <code>index</code>
	 * bezeichnete Spalte das angegebene Gewicht hinzu und dem {@link #tableWeight}
	 * auch.
	 * 
	 * @param index  Nummer der Spalte
	 * @param weight Gewicht, also Zeichenanzahl als <code>int</code>
	 * @return das erhöhte {@link #columnWeight}
	 */
	private int gainWeight(int index, int weight) {
		columnWeight[index] = columnWeight[index] + weight;
		tableWeight += weight;
		return columnWeight[index];
	}

}
