/*
 * GesamtschuldPruefer.java
 * eu.gronos.kostenrechner.util.baumbach (Kostenrechner)
 */
package eu.gronos.kostenrechner.util.baumbach;

import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.Casus.GENITIV;

import eu.gronos.kostenrechner.data.baumbach.BaumbachGesamtschuldnerschaft;
import eu.gronos.kostenrechner.interfaces.Pruefend;

/**
 * Der {@link Pruefend} kontrolliert alle Werte zum Bau einer
 * {@link BaumbachGesamtschuldnerschaft}
 *
 * @author Peter Schuster (setrok)
 * @date 16.07.2020
 *
 */
public class GesamtschuldPruefer implements Pruefend<BaumbachGesamtschuldnerschaft> {

	private static final String FEHLER_UNTERLIEGEN_GROESSER_STREITWERT = "Die gesamtschuldnerische Verurteilung (%,.2f EUR) darf zusammen mit der Verurteilung %s %ndarüber hinaus (%,.2f EUR) den Streitwert (%,.2f EUR) nicht überschreiten!";
	private static final String FEHLER_NIX_GENUG_BETEILIGTE = "Nicht genug Beteiligte ausgewählt. Wählen Sie mindestens zwei Beteiligte aus!";

	/**
	 * Die Methode wird aufgerufen, um zu prüfen, ob der Rückgabewert gültige
	 * Teilwerte hat, die zueinander nicht im Widerspruch stehen.
	 * 
	 * Hier wirft die Methode Fehler, wenn weniger als zwei Beteiligte ausgewählt
	 * sind oder wenn das Gesamtunterliegen bei einem Beteiligten größer als der
	 * Streitwert ist.
	 * 
	 * @param eingabe die zu prüfende {@link BaumbachGesamtschuldnerschaft}
	 * @return TODO
	 * @throws IllegalArgumentException TODO
	 * @throws NumberFormatException    TODO
	 * @throws NullPointerException     TODO
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Pruefend#pruefeEingabe(java.lang.Object)
	 */
	@Override
	public boolean pruefeEingabe(BaumbachGesamtschuldnerschaft eingabe)
			throws IllegalArgumentException, NumberFormatException, NullPointerException {

		if (eingabe.getAufzaehlung() == null || eingabe.getAufzaehlung().length < 2) {
			throw new NullPointerException(FEHLER_NIX_GENUG_BETEILIGTE);
		}
		for (int index : eingabe.getAufzaehlung()) {
			// Gesamtschuldnerschaften dürfen nicht ausgewählt sein
			if (eingabe.getBaumbachBeteiligtenListe().get(index) instanceof BaumbachGesamtschuldnerschaft) {
				throw new IllegalArgumentException("Sie dürfen keine Gesamtschuldnerschaften auswählen!");
			}
			// Das Unterliegen darf zusammengenommen nicht den Streitwert überschreiten.
			// if ((eingabe.getErfolg() +
			// eingabe.getBaumbachBeteiligtenListe().get(index).getErfolg()) > eingabe
			// .getBaumbachBeteiligtenListe().get(index).getAntrag()) { .add( .greaterThan(
			if ((eingabe.getErfolg().add(eingabe.getBaumbachBeteiligtenListe().get(index).getErfolg()))
					.greaterThan(eingabe.getBaumbachBeteiligtenListe().get(index).getAntrag())) {
				throw new IllegalArgumentException(String.format(FEHLER_UNTERLIEGEN_GROESSER_STREITWERT,
						eingabe.getErfolg(), eingabe.getBaumbachBeteiligtenListe().parteiBezeichner(GENITIV, index),
						eingabe.getBaumbachBeteiligtenListe().get(index).getErfolg(),
						eingabe.getBaumbachBeteiligtenListe().get(index).getAntrag()));
			}
		}
		return true;
	}

}
