/*
 * StreitgenossenPruefer.java
 * eu.gronos.kostenrechner.controller.baumbach (Kostenrechner)
 */
package eu.gronos.kostenrechner.util.baumbach;

import javax.swing.JFormattedTextField;

import eu.gronos.kostenrechner.controller.NumberExtractor;
import eu.gronos.kostenrechner.data.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.data.baumbach.BaumbachBeteiligter;
import eu.gronos.kostenrechner.data.baumbach.BaumbachGesamtschuldnerschaft;
import eu.gronos.kostenrechner.data.baumbach.Streitgenossen;
import eu.gronos.kostenrechner.data.gebuehren.Teilklageruecknahme;
import eu.gronos.kostenrechner.interfaces.Pruefend;
import eu.gronos.kostenrechner.interfaces.UnterContainerKlasse;
import eu.gronos.kostenrechner.logic.TenorTexter;

/**
 * Ein {@link Pruefend} für die {@link UnterContainerKlasse}
 * {@link Streitgenossen}
 *
 * @author Peter Schuster (setrok)
 * @date 16.07.2020
 *
 */
public class StreitgenossenPruefer implements Pruefend<Streitgenossen> {

	/**
	 * Die Methode überprüft die {@link UnterContainerKlasse} vom Typ
	 * {@link Teilklageruecknahme}, indem er ihr übergeben wird.
	 * 
	 * @param eingabe die zu prüfende {@link UnterContainerKlasse} vom Typ
	 *                {@link Streitgenossen}
	 * @return <code>true</code>, wenn die Plausibilitäten zufrieden sind
	 * @throws NumberFormatException    wenn der {@link NumberExtractor} bei einem
	 *                                  {@link JFormattedTextField} eine solche
	 *                                  Exception wirft.
	 * @throws NullPointerException     nur zur Sicherheit...
	 * @throws IllegalArgumentException wird geworfen, wenn eine der folgenden
	 *                                  Vorgaben nicht eingehalten wurde. a) Kläger
	 *                                  und Beklagte vorhanden; b) nicht mehr als 1
	 *                                  Kläger und 1 Widerkläger; c) die Klägerliste
	 *                                  darf nur Kläger, die Beklagtenliste nur
	 *                                  Beklagte, die Drittwiderbeklagtenliste nur
	 *                                  Drittwiderbeklagte enthalten; d) wenn es
	 *                                  einen Widerkläger gibt, muss es auch
	 *                                  Widerbeklagte geben; wenn es Widerbeklagte
	 *                                  gibt, muss es auch einen Widerkläger geben.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Pruefend#pruefeEingabe(java.lang.Object)
	 */
	@Override
	public boolean pruefeEingabe(Streitgenossen eingabe)
			throws IllegalArgumentException, NumberFormatException, NullPointerException {
		// TODO auch BeteiligtenPruefer und GesamtschuldPruefer aufrufen

		/* Kläger hinzufügen (wenn vorhanden), sonst Fehler */
		if (eingabe.klaeger == null || eingabe.klaeger.isEmpty())
			throw new IllegalArgumentException(TenorTexter.FEHLER_NIX_PARTEIEN);
		/* Beklagte hinzufügen, wenn vorhanden, sonst Fehler */
		if (eingabe.beklagte == null || eingabe.beklagte.isEmpty())
			throw new IllegalArgumentException(TenorTexter.FEHLER_NIX_PARTEIEN);

		/* Nur einen Kläger und nur einen Widerkläger! */
		if (eingabe.klaeger.zaehleKlaeger() > 1 || eingabe.beklagte.zaehleAnWiderklageBeteiligte() > 1)
			throw new IllegalArgumentException(TenorTexter.FEHLER_NUR_EINER);

		/*
		 * Wenn die Listen Beteiligte von einem unpassenden Typ enthalten, kann die
		 * Reihenfolge nicht eingehalten werden.
		 */
		if (eingabe.klaeger.zaehleBeklagte() > 0 || eingabe.klaeger.zaehleDrittwiderbeklagte() > 0
				|| eingabe.beklagte.zaehleKlaeger() > 0 || eingabe.beklagte.zaehleDrittwiderbeklagte() > 0
				|| eingabe.drittwiderbeklagte.zaehleKlaeger() > 0 || eingabe.drittwiderbeklagte.zaehleBeklagte() > 0)
			throw new IllegalArgumentException(TenorTexter.FEHLER_REIHENFOLGE_BETEILIGTE);

		/*
		 * Wenn es einen Widerkläger gibt, muss es mindestens einen Widerbeklagten geben
		 */
		if (eingabe.beklagte.zaehleAnWiderklageBeteiligte() > 0 && (eingabe.klaeger.zaehleAnWiderklageBeteiligte()
				+ eingabe.drittwiderbeklagte.zaehleAnWiderklageBeteiligte()) <= 0)
			throw new IllegalArgumentException(TenorTexter.FEHLER_NIX_WIDERBEKLAGTE);
		/* Wenn es Widerbeklagte gibt, muss es auch einen Widerkläger geben */
		if (eingabe.beklagte.zaehleAnWiderklageBeteiligte() <= 0 && (eingabe.klaeger.zaehleAnWiderklageBeteiligte()
				+ eingabe.drittwiderbeklagte.zaehleAnWiderklageBeteiligte()) > 0)
			throw new IllegalArgumentException(TenorTexter.FEHLER_NIX_WIDERKLAEGER);

		/* jetzt noch die Einzelkomponenten prüfen */
		final BeteiligtenPruefer betPruefer = new BeteiligtenPruefer();
		final GesamtschuldPruefer gesPruefer = new GesamtschuldPruefer();
		for (BaumbachBeteiligtenListe liste : new BaumbachBeteiligtenListe[] { eingabe.klaeger, eingabe.beklagte,
				eingabe.drittwiderbeklagte }) {
			for (BaumbachBeteiligter beteiligter : liste) {
				if (beteiligter instanceof BaumbachGesamtschuldnerschaft)
					gesPruefer.pruefeEingabe((BaumbachGesamtschuldnerschaft) beteiligter);
				else
					betPruefer.pruefeEingabe(beteiligter);
			}
		}

		return true;
	}

}
