/**
 * DatenXmlLadenSpeichern.java
 * eu.gronos.kostenrechner.util (Kostenrechner)
 */
package eu.gronos.kostenrechner.util.files;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.StringWriter;
import java.nio.file.Files;
import java.nio.file.Path;

import javax.xml.bind.JAXB;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.PropertyException;
import javax.xml.bind.Unmarshaller;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.files.KostenFileFilter;
import eu.gronos.kostenrechner.controller.files.XmlTransferHandler;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.data.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.data.baumbach.BaumbachGesamtschuldnerschaft;
import eu.gronos.kostenrechner.data.baumbach.Streitgenossen;
import eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer;

/**
 * Diese Klasse verwaltet die {@link Marshaller}s, die zum Speichern der
 * XML-Datei (zu {@link VerfahrensDatenContainer}n) benötigt werden.
 * 
 * Dazu stellt sie die Methoden für Dateioperationen
 * {@link #speicherXml(VerfahrensDatenContainer, File)} und
 * {@link #ladeXml(File)} zur Verfügung. Zu Debug-Zweck gibt es außerdem noch
 * {@link #toXmlString(VerfahrensDatenContainer)}, der statt in eine Datei in
 * einen {@link StringWriter} schreibt und einen {@link String} zurückgibt.
 *
 * @author Peter Schuster (setrok)
 * @date 01.07.2018
 *
 */
public class BerechnungXmlDatei extends XmlDatei<VerfahrensDatenContainer> {

	protected static final String FEHLER_BEIM_LADEN_DER_XML_DATEI = "Fehler beim Laden der XML-Datei.";
	public static final String DATEIENDUNG = "skktx";
	public static final String DATEITYP = "Schusters kleiner Kostentenor XML-Dateien";
	private static final String SCHEMA_LOCATION = "https://www.kostentenor.de schema-skktx.xsd";
	public static final KostenFileFilter DATEI_FILTER = new KostenFileFilter("berechnung",
			String.format("%s (%s)", DATEITYP, DATEIENDUNG), DATEIENDUNG);

	public BerechnungXmlDatei() {
		super();
	}

	/**
	 * Die Methode dient zum Umwandeln der Java Objekte (dargereicht als
	 * {@link VerfahrensDatenContainer}) in XML. In
	 * {@link VerfahrensDatenContainer#allgemein} wird das Speicherdatum auf die
	 * aktuelle Zeit gesetzt.
	 * 
	 * 
	 * @param container der {@link VerfahrensDatenContainer} mit allen zu
	 *                  speichernden Zuständen
	 * @param datei     ein {@link File}, in das zu speichern ist
	 * @throws JAXBException
	 * @throws IOException
	 */
	@Override
	public void speicherXml(VerfahrensDatenContainer container, Path datei) throws JAXBException, IOException {
		container.allgemein.speicherdatum = new XmlTransferHandler().speicherDatum();
		Marshaller marshaller = BerechnungXmlDatei.createMarshaller(true, getSchemaLocation());
//		marshaller.marshal(container, datei);
		try (OutputStream out = Files.newOutputStream(datei)) {
			marshaller.marshal(container, out);
		}
	}

	/**
	 * Die Methode liest über {@link JAXB#unmarshal(File, Class)} aus der
	 * ausgewählten Datei einen {@link VerfahrensDatenContainer} aus.
	 * 
	 * @param datei ein {@link File}, das zu laden ist
	 * 
	 * @return ein {@link VerfahrensDatenContainer} mit allen gespeicherten
	 *         Zuständen
	 * @see JAXB#unmarshal(File, Class)
	 */
	public VerfahrensDatenContainer ladeXml(Path datei) {
		VerfahrensDatenContainer container = null;
		try {
			container = unmarshalXml(datei);
			Kostenrechner.getLogger()
					.info(String.format("Geladen als vdc: %s (%s)", datei.toAbsolutePath(), container.getClass()));
		} catch (JAXBException | NumberFormatException | IOException error) {
			FehlerHelper.zeigeFehler(FEHLER_BEIM_LADEN_DER_XML_DATEI, error);
			container = null;
		}
		if (container != null) {
			container.streitgenossen = fixGesamtschuldnerschaften(container.streitgenossen);
		}
		return container;
	}

	/**
	 * Die Methode dient dazu, die Programmzustände ohne Datei in einen String zu
	 * konvertieren.
	 * 
	 * @param container der {@link VerfahrensDatenContainer} mit allen zu
	 *                  speichernden Zuständen
	 * @return einen String mit den XML-Daten oder einen Fehlertext, falls die
	 *         Konvertiertung nicht klappt
	 * @see XmlDatei#toXmlString(Object)
	 */
	@Override
	public String toXmlString(VerfahrensDatenContainer container) {
		StringWriter sw = new StringWriter();
		try {
			/* für den XML-String sollen keine Zeilenumbrüche erstellt werden */
			Marshaller marshaller = createMarshaller(false, getSchemaLocation());
			marshaller.marshal(container, sw);
		} catch (JAXBException e) {
			return "Programmzustände konnten nicht in XML konvertiert werden: " + e.getLocalizedMessage();
		}
		return sw.toString();
	}

	/**
	 * 
	 * @see eu.gronos.kostenrechner.util.files.XmlDatei#getSchemaLocation()
	 */
	@Override
	public String getSchemaLocation() {
		return SCHEMA_LOCATION;
	}

	@Override
	public String getDateiendung() {
		return DATEIENDUNG;
	}

	/**
	 * Die Methode, die den Container ausliest, muss bei
	 * {@link BaumbachGesamtschuldnerschaft}en die richtige
	 * {@link BaumbachBeteiligtenListe} hinzufügen
	 * 
	 * @param streitgenossen ein Untercontainer {@link Streitgenossen}
	 * 
	 * @return den reparierten Untercontainer {@link Streitgenossen}
	 */
	protected Streitgenossen fixGesamtschuldnerschaften(Streitgenossen streitgenossen) {
		BaumbachBeteiligtenListe beklagte = streitgenossen.beklagte;
		if (beklagte.enthaeltGesamtschuldner())
			for (int index = 0; index < beklagte.size(); index++) {
				if (beklagte.get(index) instanceof BaumbachGesamtschuldnerschaft)
					((BaumbachGesamtschuldnerschaft) beklagte.get(index)).setBaumbachBeteiligtenListe(beklagte);
			}
		BaumbachBeteiligtenListe widerbeklagte = new BaumbachBeteiligtenListe();
		widerbeklagte.addAll(streitgenossen.klaeger);
		widerbeklagte.addAll(streitgenossen.drittwiderbeklagte);
		if (widerbeklagte.enthaeltGesamtschuldner())
			for (int index = 0; index < widerbeklagte.size(); index++) {
				if (widerbeklagte.get(index) instanceof BaumbachGesamtschuldnerschaft)
					((BaumbachGesamtschuldnerschaft) widerbeklagte.get(index))
							.setBaumbachBeteiligtenListe(widerbeklagte);
			}
		return streitgenossen;
	}

	/**
	 * Die Methode dient zum Umwandeln aus XML in einen
	 * {@link VerfahrensDatenContainer}.
	 * 
	 * Die Methode muss <code>private</code> sein, damit {@link #ladeXml(File)}
	 * nicht versehentlich {@link TenorXmlDatei#unmarshalXml(File)} nimmt. Daher
	 * muss man vor dem Aufruf dieser Methode prüfen, ob die Datei eine
	 * {@link TenorXmlDatei} oder eine {@link BerechnungXmlDatei} ist!
	 * 
	 * @param datei ein {@link File}, das zu laden ist
	 * @return ein {@link VerfahrensDatenContainer} mit allen gespeicherten
	 *         Zuständen
	 * @throws JAXBException Exception that represents a failure in a JAXB
	 *                       operation.
	 * @throws IOException
	 * 
	 * @see javax.xml.bind.JAXB#unmarshal(File,Class)
	 */
	private VerfahrensDatenContainer unmarshalXml(Path datei) throws JAXBException, IOException {
		JAXBContext context = JAXBContext.newInstance(VerfahrensDatenContainer.class);
		Unmarshaller u = context.createUnmarshaller();
		try (InputStream in = Files.newInputStream(datei)) {
			return (VerfahrensDatenContainer) u.unmarshal(in);
		}
	}
//	private VerfahrensDatenContainer unmarshalXml(Path datei) throws JAXBException {
//	return JAXB.unmarshal(datei, VerfahrensDatenContainer.class);
//}

	/**
	 * Die Methode dient dazu, einen {@link Marshaller} zu erstellen, der dann zum
	 * Speichern als XML-Datei benötigt wird.
	 * 
	 * @param formattedOutput schaltet, ob der Marshaller Zeilenumbrüche und
	 *                        Einrückungen bauen soll
	 *                        {@link Marshaller#JAXB_FORMATTED_OUTPUT}
	 * @param schemaLocation  {@link #getSchemaLocation()}
	 * @return einen {@link Marshaller}
	 * @throws JAXBException
	 * @throws PropertyException
	 */
	private static Marshaller createMarshaller(boolean formattedOutput, String schemaLocation)
			throws JAXBException, PropertyException {
		JAXBContext context = JAXBContext.newInstance(VerfahrensDatenContainer.class);
		Marshaller marshaller = context.createMarshaller();
		marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, new Boolean(formattedOutput));
		marshaller.setProperty(Marshaller.JAXB_SCHEMA_LOCATION, schemaLocation);
		return marshaller;
	}
}
