/*
 * RTFWriter.java
 * eu.gronos.kostenrechner.controller.files (Kostenrechner)
 */
package eu.gronos.kostenrechner.util.files;

import java.io.UnsupportedEncodingException;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;
import java.util.Objects;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.files.RtfWerkzeugKasten;
import eu.gronos.kostenrechner.data.tenordaten.CaretRange;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.Fraction;
import eu.gronos.kostenrechner.interfaces.HtmlRtfFormattierend;
import eu.gronos.kostenrechner.interfaces.Tabulierend;
import eu.gronos.kostenrechner.interfaces.Tenorierend;

/**
 * Klasse, die für {@link RtfWerkzeugKasten} den RTF {@link String} erzeugt.
 *
 * @author Peter Schuster (setrok)
 * @date 18.08.2020
 *
 */
public class RTFStringCreator {

	private String title;
	private String gruendeUeberschrift;
	private String tenor;
	private String gruende;
	private String arial;
	private static final String RTF_CREATIM_FORMAT = "{\\creatim\\yr%d\\mo%d\\dy%d\\hr%d\\min%d}%n";
	private static final String RTF_DOCCOMM_START = "{\\doccomm\n";
	private static final String RTF_TITLE_START = "{\\title\n";
	private static final String RTF_TAG_END = "}\n";
	private static final String RTF_INFO_START = "{\\info\n";
	private static final String RTF_FONT_DEF_START = "\\deff0{\\fonttbl{\\f0\\fswiss ";
	private static final String RTF_FONT_DEF_END = ";}}\n";
	private static final String RTF_STYLESHEET_START = "{\\stylesheet";
	private static final String RTF_STYLE_START = "{\\s";
	private static final String RTF_STYLE_BASEDON = "\\sbasedon0 ";
	private static final String RTF_STYLE_END = ";}";
	private static final String RTF_TABLE_ROW_START = "{\\trowd\\trgaph180";
	private static final String RTF_CELL_WIDTH_TAG = "\\cellx";
	private static final String RTF_TABLE_ROW_END = "\\row}\n";
	private static final String RTF_TABLE_CELL_START = "{\\pard\\intbl";
	private static final String RTF_TABLE_CELL_END = "\\cell}\n";
	private static final String RTF_BODY_START = "\\deflang1031\\plain\\fs24\\widowctrl\\hyphauto\\ftnbj";
	private static final String RTF_BASE_STYLE = "\\sa142\\sl320\\slmult1\\f0\\fs24\\qj";
	private static final String RTF_TABLE_STYLE = "\\sl320\\slmult1\\f0\\fs20\\qr";
	private static final String RTF_TABLE_HEADING_STYLE = "\\sl320\\slmult1\\f0\\fs20\\qc\\b";
	private static final String RTF_LINE_INDENT = "\\li1133";
	private static final String RTF_TENOR_STYLE = RTF_LINE_INDENT + RTF_BASE_STYLE;
	private static final String RTF_BOLD_CENTERED = "\\keepn\\qc\\b";
	private static final String RTF_HEADING_STYLE = RTF_BASE_STYLE + RTF_BOLD_CENTERED;
	private static final String RTF_PARAGRAPH_START = "\n{\\pard";
	private static final String RTF_PARAGRAPH_END = "\\par}";
	private static final String RTF_TAB = "\\\\tab ";
	private static final int RTF_MAX_TABLE_WIDTH_TWIPS = 9600;
	private final boolean alsBruch;

	/**
	 * @param alsBruch
	 */
	public RTFStringCreator(boolean alsBruch) {
		super();
		this.alsBruch = alsBruch;
	}

	/**
	 * Die Methode schreibt den RTF-Inhalt des StyledDocument in einen
	 * ByteArrayOutputStream.
	 * 
	 * @param tenorierend TODO
	 * @return den Inhalt des ByteArrayOutputStream mittels toString()
	 * @throws UnsupportedEncodingException
	 * @fixed: auch hier funktionierte der Zeichensatz nicht, jetzt schleust die
	 *         Methoden den {@link String} durch
	 *         {@link #unicodeReplace(String, Hashtable)} und gibt ihn erst dann
	 *         zurück.
	 */ // , boolean alsBruch
	public String createRTFString(Tenorierend tenorierend) throws UnsupportedEncodingException {
		final StringBuilder rtfBuilder = createRtfStart();
		appendRtfHeader(rtfBuilder);// baseFont().getFamily(),
		appendRtfBody(rtfBuilder);
		// wenn Begründung, dann reasons
		if (tenorierend != null && tenorierend instanceof Tabulierend) {
			appendRtfReasons((Tabulierend) tenorierend, rtfBuilder);
		} // , alsBruch
		final String str = closeRtfBuilder(rtfBuilder);
		final Map<String, String> dict = createDictionary(str);
		return unicodeReplace(str, dict);
	}

	/**
	 * Die Methode erzeugt einen {@link StringBuilder} zur RTF-Erzeugung
	 * 
	 * @return den durchgereichten {@link StringBuilder}
	 */
	private StringBuilder createRtfStart() {
		// Start-Klammer, Magic number + encoding
		// Nach der Klammer darf kein Zeilenumbruch sein, sonst akzeptiert Word die
		// Datei nicht als RTF.
		return new StringBuilder("{\\rtf1\\ansi\n");
	}

	/**
	 * Die Methode erzeugt den RTF-Header und hängt ihn an den {@link StringBuilder}
	 * an. Die Methode fügt Metadaten ein, wie etwa den Titel des Dokuments
	 * 
	 * @param rtfBuilder den durchgereichten {@link StringBuilder}
	 * 
	 * @return den durchgereichten {@link StringBuilder}
	 */
	private StringBuilder appendRtfHeader(StringBuilder rtfBuilder) {
		rtfBuilder.append(RTF_FONT_DEF_START + arial + RTF_FONT_DEF_END);

		// Styles
		rtfBuilder.append(RTF_STYLESHEET_START + "\n");
		rtfBuilder.append(RTF_STYLE_START + "0" + " " + "Standard" + RTF_STYLE_END);
		rtfBuilder.append(RTF_STYLE_START + "1" + RTF_STYLE_BASEDON + "Tenor" + RTF_STYLE_END);
		rtfBuilder.append(RTF_STYLE_START + "2" + RTF_STYLE_BASEDON + "Ü berschrift 1" + RTF_STYLE_END);
		rtfBuilder.append(RTF_STYLE_START + "3" + RTF_STYLE_BASEDON + "Tabellenzelle" + RTF_STYLE_END);
		rtfBuilder.append(RTF_STYLE_START + "4" + RTF_STYLE_BASEDON + "Tabellenkopf" + RTF_STYLE_END);
		rtfBuilder.append(RTF_TAG_END);

		// Infos: Title usw
		final String title = this.title.replaceAll("\\\\", " ");

		rtfBuilder.append(RTF_INFO_START);
		rtfBuilder.append(RTF_TITLE_START + title + RTF_TAG_END);

		// {\creatim\yr1969\mo7\dy20\hr22\min56}
		LocalDateTime jetzt = LocalDateTime.now();
		rtfBuilder.append(String.format(RTF_CREATIM_FORMAT, jetzt.getYear(), jetzt.getMonthValue(),
				jetzt.getDayOfMonth(), jetzt.getHour(), jetzt.getMinute()));

		rtfBuilder.append(RTF_DOCCOMM_START + Kostenrechner.getInstance().getTitle() + RTF_TAG_END);
		rtfBuilder.append(RTF_TAG_END);

		return rtfBuilder;
	}

	/**
	 * Die Methode dient hängt den RTF-"Body" an den {@link StringBuilder} an, also
	 * den eingerückten Tenor-Text
	 * 
	 * @param rtfBuilder den durchgereichten {@link StringBuilder}
	 * @return den durchgereichten {@link StringBuilder}
	 */
	private StringBuilder appendRtfBody(StringBuilder rtfBuilder) {
		// 1 TWIP = 1/20 Pt. ("TWentIeth of a Point") = 1/1440 Zoll = 0,01763889 mm
		// (1/57 mm), 1mm=56,6929133858twips,

		// Los geht's mit dem RTF-"Body"
		rtfBuilder.append("\n" + RTF_BODY_START + "\n");
		rtfBuilder.append("{\\header\\pard\\plain " + RTF_BASE_STYLE + "\\qr\\chpgn" + RTF_PARAGRAPH_END + "\n");

		// Tenor nach Absätzen auftrennen und als Absätze einfügen
		appendParagraphs(rtfBuilder, tenor, "\\s1" + RTF_TENOR_STYLE);

		return rtfBuilder;
	}

	/**
	 * Die Methode hängt die Begründung an den {@link StringBuilder} an.
	 * 
	 * @param rtfBuilder den durchgereichten {@link StringBuilder}
	 * @return den durchgereichten {@link StringBuilder}
	 */ // , boolean alsBruch
	private StringBuilder appendRtfReasons(final Tabulierend tabulierend, StringBuilder rtfBuilder) {
		// Aus dem tenorierend nehmen wir nicht den Text, aber die Zeilen und den caret

		rtfBuilder.append(RTF_PARAGRAPH_START + "\\s2" + RTF_HEADING_STYLE + "\n");
		rtfBuilder.append(this.getGruendeUeberschrift());
		rtfBuilder.append(RTF_PARAGRAPH_END);

		// Tabelle aus Gründen heraustrennen mit caret
		final CaretRange range = tabulierend.getRange();
		final String gruendeVorTabelle = gruende.substring(0, range.beginn);
		appendParagraphs(rtfBuilder, gruendeVorTabelle, "\\s0" + RTF_BASE_STYLE);

		// in RTF-Tabelle wandeln
		final String gruendeTabelle = gruende.substring(range.beginn, range.ende);
		appendTableCells(rtfBuilder, gruendeTabelle, "\\s3" + RTF_TABLE_STYLE, tabulierend);
		// "\\s0" + , alsBruch // RTF_BASE_STYLE

		final String gruendeNachTabelle = gruende.substring(range.ende, gruende.length());
		appendParagraphs(rtfBuilder, gruendeNachTabelle, "\\s0" + RTF_BASE_STYLE);

		return rtfBuilder;
	}

	/**
	 * Die Methode hängt einen Abschluss an den {@link StringBuilder} und gibt ihn
	 * insgesamt als {@link String} zurück.
	 * 
	 * @param rtfBuilder den durchgereichten {@link StringBuilder}
	 * @return den durchgereichten {@link StringBuilder}
	 */
	private String closeRtfBuilder(StringBuilder rtfBuilder) {
		// Zum Schluss die geschweifte Klammer schließen
		rtfBuilder.append("\n}");
		return rtfBuilder.toString();
	}

	/**
	 * Die Methode zerlegt den übergebenen {@link String} in einzelne Absätze und
	 * hängt diese an den {@link StringBuilder} an.
	 * 
	 * @param rtfBuilder den durchgereichten {@link StringBuilder}
	 * @param string     den {@link String} mit dem unformattierten Text
	 * @param formatting der RTF-Formattierungsstring für jeden Absatz
	 * @return den durchgereichten {@link StringBuilder}
	 */
	private StringBuilder appendParagraphs(StringBuilder rtfBuilder, String string, final String formatting) {
		final String[] paragraphs = string.split("\n");
		for (String paragraph : paragraphs) {
			if (paragraph != null && !paragraph.isEmpty()) {
				rtfBuilder.append(RTF_PARAGRAPH_START + formatting + "\n");
				rtfBuilder.append(paragraph.replaceAll("\t", RTF_TAB).replaceAll("§ ", "§\\\\~"));
				//
				rtfBuilder.append(RTF_PARAGRAPH_END);
			}
		}
		return rtfBuilder;
	}

	/**
	 * Die Methode dient dazu, die Begründungstabelle als RTF-Tabelle zu erstellen
	 * und an den {@link StringBuilder} anzuhängen.
	 * 
	 * @param rtfBuilder  den durchgereichten {@link StringBuilder}
	 * @param string      den {@link String} aus den {@link #gruende}n, der die
	 *                    \t-getrennte Tabelle enthält
	 * @param formatting  der RTF-Formattierungsstring für jede Zelle
	 * @param tabulierend das {@link Tabulierend}e {@link Tenorierend}
	 * @return den durchgereichten {@link StringBuilder}
	 */
	private StringBuilder appendTableCells(StringBuilder rtfBuilder, String string, String formatting,
			final Tabulierend tabulierend) {
		final String startOfRow = cellWidthString(tabulierend);
		final String headingFormatting = "\\s4" + RTF_TABLE_HEADING_STYLE;

		// Erstmal die Überschriften
		final String[] columnHeaders = tabulierend.getColumnHeaders();
		rtfBuilder.append(startOfRow).append(headingFormatting);
		for (String headerCell : columnHeaders) {
			rtfBuilder.append(RTF_TABLE_CELL_START)//
					.append(headingFormatting)//
					.append(" ")//
					.append(headerCell != null // // Aber hier muss man trotzdem gegen null prüfen
							? headerCell.replaceAll("§ ", "§\\\\~") //
							: "")// Hier muss man leere Spalten wohl zulassen, sonst fehlen welche
					.append(RTF_TABLE_CELL_END);
		}
		rtfBuilder.append(RTF_TABLE_ROW_END);

		// jetzt: Objectdaten aus dem Tabulierend holen
		final Object[][] tableCellValues = tabulierend.getTableCellValues();
		for (Object[] row : tableCellValues) {
			if (row != null && row.length > 0) {
				// Jede Zeile bekommt ein \trowd \tgraph180 und so viele \cellx<n> wie Spalten
				rtfBuilder.append(startOfRow);
				// Dann die jeweilige Zeile (Tabellen-Zeilen-String) anhand des Tabs \t nach
				// Spalten aufteilen,
				final String[] cells = convertRow(row);
				// Jeweils {\pard\intbl …Formatierung……Text… \cell}
				for (int column = 0; column < cells.length; column++) {
					String cell = cells[column];
					rtfBuilder.append(RTF_TABLE_CELL_START)//
							.append((column == 0) //
									? headingFormatting // erste Spalte fett
									: formatting)//
							.append(" ")// convertCell verhindert null
							.append(cell.replaceAll("§ ", "§\\\\~"))//
							.append(RTF_TABLE_CELL_END);
				}
				// Nach allen Spalten, am Ende der Zeile: \row}
				rtfBuilder.append(RTF_TABLE_ROW_END);
			}
		}
		// z.B
		// \pard\intbl {Text der ersten Spalte}\cell \pard\intbl {Text der zweiten
		// Spalte}\cell \row }
		return rtfBuilder;
	}

	/**
	 * Konvertiert eine ganze Tabellenzeile aus Objektdaten
	 * 
	 * @param paragraph ein Array aus den Objektdaten einer Tabellenzeile
	 * @return ein Array aus {@link String}
	 */
	private String[] convertRow(Object[] paragraph) {
		final String[] cells = new String[paragraph.length];
		for (int i = 0; i < cells.length && i < paragraph.length; i++) {
			cells[i] = convertCell(paragraph[i]);
		}
		return cells;
	}

	/**
	 * Konvertiert ein einzelnes Objekt aus einer Tabellenzelle, sozusagen ein
	 * RtfRenderer
	 * 
	 * @param value die Roh-Objektdaten der Tabellenzelle als {@link Object}
	 * @return einen {@link String}
	 */
	private String convertCell(Object value) {
		if (value == null) // aus null wird leerer String
			return "";

		if (value instanceof String) {
			return (String) value;
		} else if (value instanceof Euro) {
			return ((HtmlRtfFormattierend) value).toRtfString();
		} else if (value instanceof Fraction) {
			final Fraction f = (Fraction) value;
			return alsBruch ? f.toRtfString() : f.toPercentString();
		} else if (value instanceof Number) {
			return String.format("%,.2f", ((Number) value).doubleValue());
		}

		return Objects.toString(value);
	}

	/**
	 * Die Methode baut einen {@link String} für die RTF-Tabellenzeile zusammen. Der
	 * muss jeweils die Breite jeder Spalte enthalten. Die Methode dient dazu,
	 * 
	 * @param tabulierend das {@link Tabulierend}e {@link Tenorierend}
	 * @return einen {@link String} für den Start einer Tabellenzeile, z.B.
	 *         <code>{\trowd \trgaph180 \cellx4800\cellx9600</code> + Zeilenwechsel
	 */
	private String cellWidthString(final Tabulierend tabulierend) {
		final String tableRowStart = RTF_TABLE_ROW_START;
		final StringBuilder builder = new StringBuilder(tableRowStart);
		final String cellWidthBeginning = RTF_CELL_WIDTH_TAG;

		final int averageCellWidth = RTF_MAX_TABLE_WIDTH_TWIPS / tabulierend.getColumnClasses().length;

		int rightCellPos = 0;
		// Dabei eine for-Schleife zum Erstellen der Zeilenbreiten nutzen:
		// \cellx<cellWidth>";
		// (vom linken Rand der TABELLE bis zum rechten Rand der SPALTE
		for (int i = 0; i < tabulierend.getColumnClasses().length; i++) {
			int weightedCellWidth = (RTF_MAX_TABLE_WIDTH_TWIPS * tabulierend.getColumnWeight(i))
					/ tabulierend.getTableWeight();
			weightedCellWidth = (weightedCellWidth + averageCellWidth) / 2;
			rightCellPos += weightedCellWidth;
			builder.append(cellWidthBeginning + rightCellPos);
		}

		builder.append("\n");
		return builder.toString();
	}

	/**
	 * Die Methode übersetzt nach den Vorgaben des <code>dict</code> die
	 * verunglückten Escape-Versuche in <code>str</code>.
	 * 
	 * @param str  der zu bereinigende {@link String}
	 * @param dict eine {@link Map}&lt;{@link String}, {@link String}&gt; mit den
	 *             Ersetzungsvorgaben
	 * @return der bereinigte <code>str</code> als {@link String}
	 */
	private String unicodeReplace(String str, Map<String, String> dict) {
		String value;

		// Alle Vorkommen aller Einträge ersetzen
		for (String key : dict.keySet()) {
			value = (String) dict.get(key);
			str = str.replaceAll(key, value);
		}

		return str;

	}

	/**
	 * Die Methode liefert die Ersetzungstabelle für
	 * {@link #unicodeReplace(String, Map)}, indem sie alle über den
	 * 7-Bit-ASCII-Bereich hinausgehenden Zeichen sammelt und mitsamt einem
	 * RTF-Unicode-Ersatz-Tag-RegEx in die {@link Map} einfügt.
	 * 
	 * @param str den {@link String} mit Unicode-Zeichen
	 * @return eine Ersetzungstabelle (&quot;Dictionary&quot;) als
	 *         {@link Map}&lt;{@link String}, {@link String}>
	 */
	private Map<String, String> createDictionary(String str) {
		// HashMap als Wörterbuch für die Ersetzungswerte erzeugen
		Map<String, String> replace = new HashMap<String, String>();

		int value;
		String bit;
		String hex;
		for (int i = 0; i < str.length(); i++) {
			// ASCII-Wert abfragen
			bit = str.substring(i, i + 1);
			value = str.codePointAt(i);

			// Wenn der ASCII-Wert mehr als der 7-bit Rahmen von RTF ASCII ist, in die
			// Hashmap einfügen (Schritt 1)
			if (value > 127) {
				replace.put("(" + bit + ")", "\\\\u" + value + "\\?");
			}

			// Anscheinend werden Umlaute bereits vorher umgewandelt, aber seltsam,
			// mit "\'" + zweistelligen Hexadezimalwert,
			// deshalb umwandeln in die oben genutzte \\u + Dezimalwert + ? Syntax (Schritt
			// 2)
			if ("\\".equals(bit) && (i + 3) < str.length() && "\\\'".equals(str.substring(i, i + 2))) {
				bit = str.substring(i, i + 4);// zB: "\'fc" für "ü"
				hex = str.substring(i + 2, i + 4);// der zweitstellige Hex-Wert, zB "fc"
				try {
					value = Integer.parseInt(hex, 16);
					replace.put("\\\\" + "\\\'" + hex, "\\\\" + "u" + value + "?");
					// z.B.: replace.put("\\\\" + "\\\'" + "fc", "\\\\" + "u" + 252 + "?");
				} catch (NumberFormatException e) {
				}
			}
		}

		return replace;
	}

	/**
	 * @param tenor d. {@link #tenor}, d. gesetzt werden soll als {@link String}.
	 */
	public void setTenor(String tenor) {
		this.tenor = tenor;
	}

	/**
	 * @param title d. {@link #title}, d. gesetzt werden soll als {@link String}.
	 */
	public void setTitle(String title) {
		this.title = title;
	}

	/**
	 * @param gruende d. {@link #gruende}, d. gesetzt werden soll als
	 *                {@link String}.
	 */
	public void setGruende(String gruende) {
		this.gruende = gruende;
	}

	/**
	 * @param gruendeUeberschrift d. {@link #gruendeUeberschrift}, d. gesetzt werden
	 *                            soll als {@link String}.
	 */
	public void setGruendeUeberschrift(String gruendeUeberschrift) {
		this.gruendeUeberschrift = gruendeUeberschrift;
	}

	/**
	 * @return gibt {@link #gruendeUeberschrift} als {@link String} zurück.
	 */
	public String getGruendeUeberschrift() {
		return gruendeUeberschrift;
	}

	/**
	 * @param arial d. {@link #arial}, d. gesetzt werden soll als {@link String}.
	 */
	public void setArial(String arial) {
		this.arial = arial;
	}
}