/**
 * XJustizErrorHandler.java
 * eu.gronos.kostenrechner.controller (Kostenrechner)
 */
package eu.gronos.kostenrechner.util.files;

import java.util.logging.Level;

import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import com.sun.org.apache.xerces.internal.parsers.XMLParser;

import eu.gronos.kostenrechner.Kostenrechner;

/**
 * Baut aus den Fehlermeldungen des {@link XMLParser} durch Hinzufügen von etwas
 * mehr Information schönere Exceptions
 *
 * @author Peter Schuster (setrok)
 * @date 10.08.2014
 *
 */
public class XJustizErrorHandler implements ErrorHandler {

	/**
	 * Leerer Konstruktor, was will man mehr?
	 */
	public XJustizErrorHandler() {
		super();
	}

	/**
	 * Baut aus den Fehlermeldungen des {@link XMLParser} durch Hinzufügen von etwas
	 * mehr Information schönere Exceptions
	 * 
	 * @param spe eine {@link SAXParseException}
	 * @return ein String, bei dem der LocalizedMessage und Message auch URI und
	 *         Zeilennummer zugefügt sind
	 */
	private String getParseExceptionInfo(SAXParseException spe) {
		String systemId = spe.getSystemId();
		if (systemId == null) {
			systemId = "null";
		}
		String info = "URI=" + systemId + "Line=" + spe.getLineNumber() + ":" + spe.getLocalizedMessage() + " ("
				+ spe.getMessage() + ")";
		return info;
	}

	/**
	 * Receive notification of a warning.
	 * 
	 * SAX parsers will use this method to report conditions that are not errors or
	 * fatal errors as defined by the XML recommendation. The default behaviour is
	 * to take no action.
	 * 
	 * The SAX parser must continue to provide normal parsing events after invoking
	 * this method: it should still be possible for the application to process the
	 * document through to the end.
	 * 
	 * Filters may use this method to report other, non-XML warnings as well.
	 * 
	 * @param spe eine {@link SAXParseException}
	 * @throws SAXException wird nicht geworfen, weil nur Warnung
	 * 
	 * @see org.xml.sax.ErrorHandler#warning(org.xml.sax.SAXParseException)
	 */
	@Override
	public void warning(SAXParseException spe) throws SAXException {
		Kostenrechner.getLogger().log(Level.WARNING, String.format("Warnung: %s", getParseExceptionInfo(spe)), spe);
	}

	/**
	 * Receive notification of a recoverable error.
	 * 
	 * This corresponds to the definition of "error" in section 1.2 of the W3C XML
	 * 1.0 Recommendation. For example, a validating parser would use this callback
	 * to report the violation of a validity constraint. The default behaviour is to
	 * take no action.
	 * 
	 * The SAX parser must continue to provide normal parsing events after invoking
	 * this method: it should still be possible for the application to process the
	 * document through to the end. If the application cannot do so, then the parser
	 * should report a fatal error even if the XML recommendation does not require
	 * it to do so.
	 * 
	 * Filters may use this method to report other, non-XML errors as well.
	 * 
	 * @param spe eine {@link SAXParseException}
	 * @throws SAXException dieselbe Exception als {@link SAXException}, erweitert
	 *                      um URI und Zeilenummer.
	 * 
	 * @see org.xml.sax.ErrorHandler#error(org.xml.sax.SAXParseException)
	 */
	@Override
	public void error(SAXParseException spe) throws SAXException {
		String message = "Error: " + getParseExceptionInfo(spe);
		// TODO Meldung
		System.err.println(message);
		throw new SAXException(message);
	}

	/**
	 * Receive notification of a non-recoverable error.
	 * 
	 * There is an apparent contradiction between the documentation for this method
	 * and the documentation for org.xml.sax.ContentHandler.endDocument. Until this
	 * ambiguity is resolved in a future major release, clients should make no
	 * assumptions about whether endDocument() will or will not be invoked when the
	 * parser has reported a fatalError() or thrown an exception.
	 * 
	 * This corresponds to the definition of "fatal error" in section 1.2 of the W3C
	 * XML 1.0 Recommendation. For example, a parser would use this callback to
	 * report the violation of a well-formedness constraint.
	 * 
	 * The application must assume that the document is unusable after the parser
	 * has invoked this method, and should continue (if at all) only for the sake of
	 * collecting additional error messages: in fact, SAX parsers are free to stop
	 * reporting any other events once this method has been invoked.
	 * 
	 * @param spe eine {@link SAXParseException}
	 * @throws SAXException dieselbe Exception als {@link SAXException}, erweitert
	 *                      um URI und Zeilenummer.
	 * 
	 * @see org.xml.sax.ErrorHandler#fatalError(org.xml.sax.SAXParseException)
	 */
	@Override
	public void fatalError(SAXParseException spe) throws SAXException {
		String message = String.format("Fatal Error: %s", getParseExceptionInfo(spe));
		Kostenrechner.getLogger().log(Level.SEVERE, message, spe);
		throw new SAXException(message);
	}
}