/*
 * GebuehrenBerechnungPruefer.java
 * eu.gronos.kostenrechner.util.gebuehren (Kostenrechner)
 */
package eu.gronos.kostenrechner.util.gebuehren;

import eu.gronos.kostenrechner.data.gebuehren.GebuehrenBerechnung;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenTatbestand;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.interfaces.Pruefend;
import eu.gronos.kostenrechner.interfaces.UnterContainerKlasse;

/**
 * Ein {@link Pruefend} für {@link GebuehrenBerechnung}
 *
 * @author Peter Schuster (setrok)
 * @date 16.07.2020
 *
 */
public class GebuehrenBerechnungPruefer implements Pruefend<GebuehrenBerechnung> {

	/**
	 * Die Methode überprüft {@link UnterContainerKlasse} vom Typ
	 * {@link GebuehrenBerechnung}, indem sie ihr übergeben wird.
	 * 
	 * @param eingabe die zu prüfende {@link UnterContainerKlasse} vom Typ
	 *                {@link GebuehrenBerechnung}
	 * @return <code>true</code>, wenn die Plausibilitäten zufrieden sind
	 * @throws NullPointerException     wenn values null ist
	 * @throws IllegalArgumentException wenn streitwert kleiner 0 ist
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Pruefend#pruefeEingabe(java.lang.Object)
	 */
	@Override
	public boolean pruefeEingabe(GebuehrenBerechnung eingabe) throws IllegalArgumentException, NullPointerException {
		// strenge Variante
		if (eingabe.streitwert.lessThan(Euro.ZERO_CENTS))// < 0
			throw new IllegalArgumentException("Der Streitwert muss größer/gleich 0 sein");

		/* Sind überhaupt GebuehrenTatbestände da? */
		if (eingabe.gebuehren == null || eingabe.gebuehren.isEmpty())
			throw new NullPointerException("Die Liste der Tatbestände darf nicht leer (null) sein");

		/* Jetzt noch die Einzelkomponenten prüfen */
		final GebuehrenPruefer pruefer = new GebuehrenPruefer();
		for (GebuehrenTatbestand gt : eingabe.gebuehren) {
			pruefer.pruefeEingabe(gt);
		}

		return true;
	}

}
