/*
 * GebuehrenPruefer.java
 * eu.gronos.kostenrechner.util.gebuehren (Kostenrechner)
 */
package eu.gronos.kostenrechner.util.gebuehren;

import eu.gronos.kostenrechner.data.gebuehren.AuslagenTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenAnrechnungsTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenErhoehungsTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenRahmenTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenSatzTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.MehrfachPauschalTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.PauschalTatbestand;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.interfaces.Pruefend;
import eu.gronos.kostenrechner.view.HinzufuegenDialog;

/**
 * Der {@link Pruefend} kontrolliert alle Werte zum Bau eines
 * {@link GebuehrenTatbestand}s
 *
 * @author Peter Schuster (setrok)
 * @date 16.07.2020
 *
 */
public class GebuehrenPruefer implements Pruefend<GebuehrenTatbestand> {

	/**
	 * Die Methode überprüft den {@link GebuehrenTatbestand}, indem er ihr übergeben
	 * wird.
	 * 
	 * @param eingabe der zu prüfende {@link GebuehrenTatbestand}
	 * @return <code>true</code>, wenn die Plausibilitäten zufrieden sind
	 * 
	 * @throws IllegalArgumentException
	 * @throws NumberFormatException
	 * @throws NullPointerException     TODO
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Pruefend#pruefeEingabe(java.lang.Object)
	 */
	@Override
	public boolean pruefeEingabe(GebuehrenTatbestand eingabe)
			throws IllegalArgumentException, NumberFormatException, NullPointerException {
		if (eingabe instanceof MehrfachPauschalTatbestand) {
			// TODO: gefährlich, wenn beim XML-Parsen geprüft wird!
			if (((MehrfachPauschalTatbestand) eingabe).getAnzahl() < 0)
				throw new NumberFormatException(HinzufuegenDialog.FEHLER_ANZAHL_NEGATIV);
		} else if (eingabe instanceof AuslagenTatbestand) {
			// TODO: gefährlich, wenn beim XML-Parsen geprüft wird!
			if (((AuslagenTatbestand) eingabe).getBetrag() == null
					|| ((AuslagenTatbestand) eingabe).getBetrag().compareTo(Euro.ofCents(0L)) < 0) {
				System.out.println("Betrag:"+((AuslagenTatbestand) eingabe).getBetrag());
				throw new NumberFormatException(HinzufuegenDialog.FEHLER_AUSLAGENHOEHE_NEGATIV);}
		} else if (eingabe instanceof PauschalTatbestand) {
			// stand vorher in GebuehrenSatzTatbestand#setSatz()
			if (((PauschalTatbestand) eingabe).getBetrag() == null
					|| ((PauschalTatbestand) eingabe).getBetrag().compareTo(Euro.ofCents(0L)) < 0) {
				throw new NumberFormatException(
						String.format("Bei einer %s darf der Betrag nicht negativ sein, hier %,.2f",
								PauschalTatbestand.ART, ((PauschalTatbestand) eingabe).getBetrag()));
			}
		} else if (eingabe instanceof GebuehrenErhoehungsTatbestand) {
			if (((GebuehrenErhoehungsTatbestand) eingabe).getAnzahl() < 0)
				throw new NumberFormatException(HinzufuegenDialog.FEHLER_ANZAHL_NEGATIV);
		} else if (eingabe instanceof GebuehrenAnrechnungsTatbestand) {
			GebuehrenAnrechnungsTatbestand grt = (GebuehrenAnrechnungsTatbestand) eingabe;
			if (Math.abs(grt.getSatz()) < Math.abs(grt.getUntergrenze())
					|| Math.abs(grt.getSatz()) > Math.abs(grt.getObergrenze())) {
				throw new NumberFormatException(
						String.format("Die angemessene Gebühr muss zwischen %,.2f- und %,.2f-fach liegen!",
								grt.getUntergrenze(), grt.getObergrenze()));
			}
		} else if (eingabe instanceof GebuehrenRahmenTatbestand) {
			GebuehrenRahmenTatbestand grt = (GebuehrenRahmenTatbestand) eingabe;
			if (grt.getSatz() < grt.getUntergrenze() || grt.getSatz() > grt.getObergrenze()) {
				// Damit wird auch geprüft, ob der Satz negativ ist
				throw new NumberFormatException(
						String.format("Die angemessene Gebühr muss zwischen %,.1f- und %,.1f-fach liegen!",
								grt.getUntergrenze(), grt.getObergrenze()));
			}
		} else if (eingabe instanceof GebuehrenSatzTatbestand) {
			// Diese Prüfung stand vorher in GebuehrenSatzTatbestand#setSatz()
			if (((GebuehrenSatzTatbestand) eingabe).getSatz() < 0) {
				throw new NumberFormatException(
						String.format("Bei einem %s darf der Satz nicht negativ sein, hier %,.2f",
								GebuehrenSatzTatbestand.ART, ((GebuehrenSatzTatbestand) eingabe).getSatz()));
			}
		}

		return true;
	}

}
