/**
 * StringConfigOptionHorcher.java
 * eu.gronos.kostenrechner.controller.system (Kostenrechner)
 */
package eu.gronos.kostenrechner.util.system;

import eu.gronos.beschriftungen.model.StringConfigSchluessel;
import eu.gronos.beschriftungen.util.PreferencesConfigOption;
import eu.gronos.kostenrechner.Kostenrechner;

/**
 * Eine von {@link PreferencesConfigOption} abgeleitete Klasse für
 * {@link String}-Werte. Vergleicht, ob in der Registry der vorgegebene Werte
 * steht.
 *
 * @author Peter Schuster (setrok)
 * @date 21 Feb 2019
 *
 */
public class LizenzConfigOption extends PreferencesConfigOption<String> {
	// private final String standardWert;

	/**
	 * 
	 * @param parameter ein {@link StringConfigSchluessel}
	 */
	public LizenzConfigOption(StringConfigSchluessel parameter) {
		super(parameter);
	}
	// String schluessel, String standardWert
	// standardWert;
	// this.standardWert =

	/**
	 * Schreibt in den Java-Zweig der Registry (bzw. der Preferences), und zwar in
	 * den Schlüssel {@link #SCHLUESSEL}, in welcher Version die Lizenzbedingungen
	 * angenommen wurden.
	 * 
	 * @param isAccepted <code>true</code>, wenn zugestimmt wurde, dann wird die
	 *                   aktuelle Version in die Registry geschrieben. Bei
	 *                   <code>false</code> wird {@link #STANDARD_WERT} in die
	 *                   Registry geschrieben.
	 * 
	 * @url {@link "http://openbook.galileocomputing.de/javainsel/javainsel_11_009.html#dodtp1f33b78d-493f-43a5-be9c-afc95d5f903b"}
	 *      Java ist auch eine Insel Kapitel 11.9.2 Einträge einfügen, auslesen und
	 *      löschen
	 */
	public void confirm(boolean isAccepted) {
		String s;
		if (isAccepted)
			s = Kostenrechner.KOSTENRECHNER_VERSION;
		else
			s = getStandardWert();
		inRegistrySchreiben(s);
	}

	/**
	 * Liest aus dem Java-Zweig der Registry, unter {@link #getSchluessel()}, ob für
	 * die aktuelle Version des Programms die Lizenz schon akzeptiert wurde.
	 * 
	 * @param wert Version des Programms als {@link String}
	 * 
	 * @return <code>true</code>, wenn in der Registry die aktuelle Version steht,
	 *         sonst <code>false</code>.
	 * @url {@link "http://openbook.galileocomputing.de/javainsel/javainsel_11_009.html#dodtp1f33b78d-493f-43a5-be9c-afc95d5f903b"}
	 *      Java ist auch eine Insel Kapitel 11.9.2 Einträge einfügen, auslesen und
	 *      löschen
	 */
	public boolean equalsRegistry(String wert) {
		if (wert == null || ausRegistryLesen() == null) {
			return false;
		}
		return wert.equals(ausRegistryLesen());
	}

	/**
	 * @return gibt {@link #standardWert} als String zurück.
	 */
	public String getStandardWert() {
		return standardWert;
	}

	/**
	 * Schreibt den übergebenen Wahrheitswert in den Java-Zweig der Registry, und
	 * zwar in den Schlüssel, der dem Konstruktor übergeben wurde.
	 * 
	 * @param wert
	 */
	protected void inRegistrySchreiben(String wert) {
		getUserprefs().put(getSchluessel(), wert);
		logInfo(wert);
	}

	/**
	 * Liest einen Wert aus dem Java-Zweig der Registry, unter dem Schluessel, der
	 * dem Konstruktor mitgegeben wurde.
	 * 
	 * @return
	 */
	protected String ausRegistryLesen() {
		return getUserprefs().get(getSchluessel(), getStandardWert());
	}

}
