/**
 * KostenTenorLadenSpeichern.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.view;

import java.awt.Component;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;

import javax.swing.JFileChooser;
import javax.xml.bind.JAXBException;

import eu.gronos.beschriftungen.model.Beschriftung;
import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.util.files.BerechnungXmlDatei;
import eu.gronos.kostenrechner.util.files.TenorXmlDatei;

/**
 * Eine Klasse, die alle Zustände des Programms im XML-Format speichern kann.
 *
 * @author Peter Schuster (setrok)
 * @date 26.12.2014
 *
 * @url "http://java.kompf.de/jaxbjava.html"
 * @url "http://docs.oracle.com/javase/tutorial/jaxb/intro/"
 */
public class BerechnungFileChooser extends KostenFileChooser {

	private static final long serialVersionUID = 5816976530887334541L;

	/**
	 * Konstruktor
	 * 
	 * @param beschriftung eine {@link Beschriftung}
	 * 
	 */
	public BerechnungFileChooser(Beschriftung beschriftung) {
		super(BerechnungXmlDatei.DATEI_FILTER, beschriftung);
	}

	/**
	 * Die Methode dient dazu, einen Dateiauswahldialog der {@link JFileChooser
	 * Oberklasse} aufzurufen um dann mit
	 * {@link BerechnungXmlDatei#speicherXml(VerfahrensDatenContainer, File)} die
	 * übergebenen Programmzustände als XML zu speichern.
	 * 
	 * @param parent    die aufrufende {@link Component}
	 * @param container der {@link VerfahrensDatenContainer} mit allen zu
	 *                  speichernden Zuständen
	 * @return den Rückgabestatus des {@link JFileChooser}s:<br>
	 *         <ul>
	 *         <li>{@link JFileChooser#CANCEL_OPTION}</li>
	 *         <li>{@link JFileChooser#APPROVE_OPTION}</li>
	 *         <li>{@link JFileChooser#ERROR_OPTION}, wenn der Dialog einen Fehler
	 *         auslöst, er abgebrochen wird oder es einen Fehler beim Laden
	 *         gibt.</li>
	 * 
	 * @see JFileChooser#showSaveDialog(Component)
	 * @see JFileChooser#showDialog(Component, String)
	 */
	public int showSaveDialog(Component parent, VerfahrensDatenContainer container) {
		int option = super.showSaveDialog(parent);
		final BerechnungXmlDatei berechnungXml = new BerechnungXmlDatei();
		if (option == JFileChooser.APPROVE_OPTION)
			try {
//				File datei = super.getSelectedFile();
				Path datei = getSelectedPath();
				berechnungXml.speicherXml(container, datei);
				Kostenrechner.getLogger().info(String.format("Gespeichert: %s", datei.toAbsolutePath()));
			} catch (JAXBException | IOException jioe) {
				option = JFileChooser.ERROR_OPTION;
				FehlerHelper.zeigeFehler("Fehler beim Speichern als XML!", jioe);
			}
		return option;
	}

	/**
	 * Die Methode dient dazu, einen Dateiauswahldialog der {@link JFileChooser
	 * Oberklasse} aufzurufen um dann mit {@link BerechnungXmlDatei#ladeXml(File)}
	 * aus der ausgewählte Datei einen {@link VerfahrensDatenContainer} auszulesen.
	 * 
	 * @param parent die aufrufende {@link Component}
	 * @return einen {@link VerfahrensDatenContainer} zum Auslesen
	 * 
	 * @see JFileChooser#showDialog(Component, String)
	 * @see javax.swing.JFileChooser#showOpenDialog(java.awt.Component)
	 */
	public VerfahrensDatenContainer ladeXmlDialog(Component parent) {
		VerfahrensDatenContainer container = null;
		int option = super.showOpenDialog(parent);
		if (option == JFileChooser.APPROVE_OPTION) {
//			File datei = super.getSelectedFile();
			Path datei = getSelectedPath();
			final BerechnungXmlDatei berechnungXml = TenorXmlDatei.createInstanceFor(datei);
			container = berechnungXml.ladeXml(datei);
		}
		return container;
	}

}
