/*
 * KostenFileChooser.java
 * eu.gronos.kostenrechner.view (Kostenrechner)
 */
package eu.gronos.kostenrechner.view;

import java.awt.Component;
import java.awt.GraphicsEnvironment;
import java.awt.HeadlessException;
import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileFilter;

import eu.gronos.beschriftungen.ComponentBeschrifter;
import eu.gronos.beschriftungen.interfaces.Vorsorgend;
import eu.gronos.beschriftungen.model.Beschriftung;
import eu.gronos.beschriftungen.model.StringConfigSchluessel;
import eu.gronos.beschriftungen.util.StringConfigOption;
import eu.gronos.kostenrechner.controller.files.KostenFileFilter;
import eu.gronos.kostenrechner.interfaces.AbfrageLieferant;

/**
 * Die Klasse dient dazu, einen {@link JFileChooser} zu erzeugen, der fürs
 * Speichern von RTF- und XML-Dateien vorkonfiguriert ist.
 *
 * @author Peter Schuster (setrok)
 * @date 9 Jun 2019
 *
 */
public class KostenFileChooser extends JFileChooser implements AbfrageLieferant<JDialog> {
	private static Path speicherOrt;// File
	private static StringConfigOption speicherOrtOption = new StringConfigOption(
			new StringConfigSchluessel("speicherOrt", ""));

	private static final long serialVersionUID = -8758824500337543760L;
	private final KostenFileFilter filter;
	private JDialog dialog2 = null;
	private ComponentBeschrifter beschrifter = new ComponentBeschrifter();
	private Beschriftung beschriftung;

	/**
	 * Erzeugt einen JFileChooser, der fürs Speichern von Dateien des übergebenen
	 * {@link FileFilter}s vorkonfiguriert ist.
	 * 
	 * @param filter       ein {@link FileFilter}
	 * @param beschriftung eine {@link Beschriftung} für den Dialog
	 */
	public KostenFileChooser(KostenFileFilter filter, Beschriftung beschriftung) {
		super();
		this.filter = filter;
		this.beschriftung = beschriftung;
	}

	/**
	 * Die Methode dient dazu, einen Dateiauswahldialog der {@link JFileChooser}
	 * Oberklasse aufzurufen. Der Speicherpfad wird gespeichert.
	 * 
	 * @param parent
	 * @return the return state of the file chooser on popdown:
	 *         <ul>
	 *         <li>JFileChooser.CANCEL_OPTION
	 *         <li>JFileChooser.APPROVE_OPTION
	 *         <li>JFileChooser.ERROR_OPTION if an error occurs or the dialog is
	 *         dismissed
	 *         </ul>
	 * 
	 * @see javax.swing.JFileChooser#showOpenDialog(java.awt.Component)
	 */
	public int showOpenDialog(Component parent) {
		setCurrentDirectory(getSpeicherOrt().toFile());
		setDialogType(OPEN_DIALOG);
		int option = showDialog(parent);
		if (option == JFileChooser.APPROVE_OPTION) {
			// jetzt den Pfad *merken*!
			setSpeicherOrt(getCurrentDirectory().toPath());
		}
		return option;
	}

	/**
	 * Die Methode poppt einen Datei-Speicher-Dialog auf. Sie kümmert sich darum,
	 * dass der eingegebene Dateiname die Endung aus
	 * {@link KostenFileChooser#KostenFileChooser(FileFilter, Beschriftung)}
	 * bekommt. Sie warnt auch, wenn eine Datei desselben Namens schon existiert und
	 * lässt den Nutzer reagieren. Der Speicherpfad wird gespeichert.
	 * 
	 * @param parent the parent component of the dialog, can be <code>null</code>;
	 *               see <code>showDialog</code> for details
	 * @return the return state of the file chooser on popdown:
	 *         <ul>
	 *         <li>JFileChooser.CANCEL_OPTION
	 *         <li>JFileChooser.APPROVE_OPTION
	 *         <li>JFileChooser.ERROR_OPTION if an error occurs or the dialog is
	 *         dismissed
	 *         </ul>
	 * 
	 * @see javax.swing.JFileChooser#showSaveDialog(java.awt.Component)
	 */
	public int showSaveDialog(Component parent) {
		setCurrentDirectory(getSpeicherOrt().toFile());
		setDialogType(SAVE_DIALOG);
		int option = showDialog(parent);
		Path datei = null;
		if (option == JFileChooser.APPROVE_OPTION) {
			datei = getSelectedPath().toAbsolutePath();
			datei = filter.acceptAsPath(datei);
			setSelectedPath(datei);
			option = pruefeVorhandensein(datei);
			// jetzt den Pfad *merken*!
			if (option == JFileChooser.APPROVE_OPTION)
				setSpeicherOrt(getCurrentDirectory().toPath());
		}
		return option;
	}

	/**
	 * @return
	 */
	public Path getSelectedPath() {
		return super.getSelectedFile().toPath();
	}

	/**
	 * @param datei
	 */
	public void setSelectedPath(Path datei) {
		super.setSelectedFile(datei.toFile());
	}

//	public int showSaveDialog(Component parent) {
//		setCurrentDirectory(getSpeicherOrt().toFile());
//		setDialogType(SAVE_DIALOG);
//		int option = showDialog(parent);
//		File datei = null;
//		if (option == JFileChooser.APPROVE_OPTION) {
//			datei = getSelectedFile().getAbsoluteFile();
//			datei = filter.pruefeEndung(datei);
//			setSelectedFile(datei);
//			option = pruefeVorhandensein(datei);
//			// jetzt den Pfad *merken*!
//			if (option == JFileChooser.APPROVE_OPTION)
//				setSpeicherOrt(getCurrentDirectory());
//		}
//		return option;
//	}

	/**
	 * @deprecated use {@link #setSelectedPath(Path)} instead
	 */
	@Override
	@Deprecated
	public void setSelectedFile(File file) {
		super.setSelectedFile(file);
	}

	/**
	 * @deprecated use {@link #getSelectedPath()} instead
	 */
	@Override
	@Deprecated
	public File getSelectedFile() {
		return super.getSelectedFile();
	}

	/**
	 * @return gibt {@link #dialog2} als {@link JDialog} zurück.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.AbfrageLieferant#getDialog()
	 */
	@Override
	public JDialog getDialog() {
		return dialog2;
	}

	/**
	 * Die Methode ruft im Prinzip nur
	 * {@link JFileChooser#createDialog(java.awt.Component)} auf, aber greift sich
	 * den {@link #getDialog()} ab. An den kommt man sonst nicht heran.
	 * 
	 * @param parent einen {@link JFrame} oder {@link JDialog}
	 * @return den erzeugten {@link JDialog}
	 * @throws HeadlessException wenn {@link GraphicsEnvironment#isHeadless()}
	 *                           <code>true</code> liefert
	 * 
	 * @see javax.swing.JFileChooser#createDialog(java.awt.Component)
	 */
	@Override
	protected JDialog createDialog(Component parent) throws HeadlessException {
		setMultiSelectionEnabled(false);
		/*
		 * Den "alle Dateien" Filter muss man wohl entfernen, bevor man den gewünschten
		 * Filter hinzufügt.
		 */
		setAcceptAllFileFilterUsed(false);
		addChoosableFileFilter(filter);
		setFileFilter(filter);
		setCurrentDirectory(getSpeicherOrt().toFile());
		if (filter.getPreSelected() != null)
			setSelectedFile(new File(filter.getPreSelected()));

		setDialog(super.createDialog(parent));
		if (beschriftung != null) {
			// Richtigen Titel wählen
			if (beschriftung instanceof Vorsorgend<?>) {
				beschrifter.changeTo((Vorsorgend<?>) beschriftung, getDialogType(), filter.getDescription());
			}
			beschrifter.beschrifte(getDialog(), beschriftung);
		}
		return getDialog();
	}

	/**
	 * @param dialog d. {@link #dialog2}, d. gesetzt werden soll als
	 *               {@link JDialog}.
	 */
	protected void setDialog(JDialog dialog) {
		this.dialog2 = dialog;
	}

	/**
	 * Die Methode gibt den {@link #speicherOrt} zurück, also das Verzeichnis, in
	 * eine Unterklasse zuletzt eine Datei gespeichert bzw. aus dem sie zuletzt eine
	 * Datei geladen hat. Liest aus Registry!
	 * 
	 * @return gibt {@link #speicherOrt} als {@link Path} zurück.
	 */
	protected Path getSpeicherOrt() {
		if (speicherOrt == null) {
			speicherOrt = Paths.get(speicherOrtOption.getValue());
			if (!Files.exists(speicherOrt) || !Files.isDirectory(speicherOrt)) {
				speicherOrtOption.reset();
				speicherOrt = Paths.get(speicherOrtOption.getValue());
			}
		}
		return speicherOrt;
	}

//	protected File getSpeicherOrtFile() {// File
//		if (speicherOrt == null) {
//			speicherOrt = new File(speicherOrtOption.getValue());
//			if (!speicherOrt.exists() || !speicherOrt.isDirectory()) {
//				speicherOrtOption.reset();
//				speicherOrt = new File(speicherOrtOption.getValue());
//			}
//		}
//		return speicherOrt;
//	}

	/**
	 * Setzt {@link #getSpeicherOrt()} und speichert die Zeichenkette in der
	 * Registry!
	 * 
	 * @param pfad d. {@link #speicherOrt}, d. gesetzt werden soll als {@link Path}.
	 */
	protected void setSpeicherOrt(Path pfad) {
		speicherOrt = pfad;
		speicherOrtOption.setValue(speicherOrt.toAbsolutePath().toString());
	}

//	protected void setSpeicherOrt(File pfad) {
//		speicherOrt = pfad;
//		speicherOrtOption.setValue(speicherOrt.getAbsolutePath());
//	}

	/**
	 * Die Methode ruft {@link JFileChooser#showDialog(Component, String)} auf, aber
	 * setzt {@link JFileChooser#getApproveButtonText()} auf <code>null</code>.
	 * 
	 * Setzt nach dem Aufruf auf {@link #getDialog()} wieder auf <code>null</code>.
	 * 
	 * @param parent einen {@link JFrame} oder {@link JDialog}
	 * @return {@link JFileChooser#CANCEL_OPTION}, {@link JFileChooser#ERROR_OPTION}
	 *         oder {@link JFileChooser#APPROVE_OPTION}
	 * 
	 * @see javax.swing.JFileChooser#showDialog(Component, String)
	 */
	private int showDialog(Component parent) {
		int option = showDialog(parent, null);
		setDialog(null);
		return option;
	}

	/**
	 * Die Methode warnt, wenn eine Datei desselben Namens schon existiert und lässt
	 * den Nutzer reagieren.
	 * 
	 * @param datei die gewählte Datei als {@link Path}
	 * 
	 * @return the return state of the file chooser on popdown:
	 *         <ul>
	 *         <li>JFileChooser.CANCEL_OPTION wenn die Datei schon existiert und der
	 *         Nutzer sie nicht überschreiben will
	 *         <li>JFileChooser.APPROVE_OPTION wenn die Datei noch nicht existiert
	 *         oder der Nutzer einverstanden ist sie zu überschreiben
	 *         </ul>
	 */
	private int pruefeVorhandensein(Path datei) {
		if (Files.exists(datei)) {
			int bestaetigung = JOptionPane.showConfirmDialog(this, "Die Datei existiert schon. \n\nFortfahren?");
			if (bestaetigung == JOptionPane.OK_OPTION) {
				System.out.println("Zugestimmt");
				return JFileChooser.APPROVE_OPTION;
			} else {
				JOptionPane.showMessageDialog(this, "Die Datei wurde nicht gespeichert.");
				return JFileChooser.CANCEL_OPTION;
			}
		}
		System.out.println("keine Zustimmung nötig");
		return JFileChooser.APPROVE_OPTION;
	}

//	private int pruefeVorhandensein(File datei) {
//		if (datei.exists()) {
//			int bestaetigung = JOptionPane.showConfirmDialog(this, "Die Datei existiert schon. \n\nFortfahren?");
//			if (bestaetigung == JOptionPane.OK_OPTION) {
//				System.out.println("Zugestimmt");
//				return JFileChooser.APPROVE_OPTION;
//			} else {
//				JOptionPane.showMessageDialog(this, "Die Datei wurde nicht gespeichert.");
//				return JFileChooser.CANCEL_OPTION;
//			}
//		}
//		System.out.println("Zugestimmt");
//		return JFileChooser.APPROVE_OPTION;
//	}

}
