/**
 * BaumbachPanel.java
 * eu.gronos.kostenrechner.view (Kostenrechner)
 */
package eu.gronos.kostenrechner.view.baumbach;

import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp.BEKLAGTE;
import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp.DRITTWIDERBEKLAGTE;
import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp.KLAEGER;

import java.awt.GridBagLayout;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;

import eu.gronos.beschriftungen.controller.BeschriebeneAktion;
import eu.gronos.beschriftungen.model.Beschriftung;
import eu.gronos.beschriftungen.model.LangBeschriftung;
import eu.gronos.beschriftungen.model.NameContainerSammlung;
import eu.gronos.beschriftungen.view.PopupMenuBuilder;
import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.EintragEntfernenAktion;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenAssistentAction;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenBaumbachAction;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenBeteiligterNeuAction;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenGesamtschuldBkExtractAction;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenGesamtschuldDwbkExtractAction;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenGesamtschuldnerBkAction;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenGesamtschuldnerDwbkAction;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenParser;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenWiderklageAction;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenXJustizLesenAction;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.data.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.data.baumbach.Streitgenossen;
import eu.gronos.kostenrechner.data.baumbach.Widerklage;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.interfaces.StreitgenossenTabellenBehandelnd;
import eu.gronos.kostenrechner.interfaces.WerteLesbar;
import eu.gronos.kostenrechner.model.baumbach.BeteiligtenTableModel;
import eu.gronos.kostenrechner.view.KostenJPanel;

/**
 * Diese von {@link KostenJPanel} abgeleitete Klasse nimmt die
 * Oberflächenelemente für die Berechnung der Baumbachschen Formel auf.
 *
 * @author Peter Schuster (setrok)
 * @date 13.06.2018
 *
 */
public class StreitgenossenPanel extends KostenJPanel
		implements WerteLesbar<Streitgenossen>, StreitgenossenTabellenBehandelnd {

	private static final long serialVersionUID = 1265694381131490890L;
	private static final StreitgenossenPanel instance = new StreitgenossenPanel();
	private Kostenrechner kostenrechner;
	private boolean widerKlageAnzeigen;
	public JTable tabKlaeger;
	public BeteiligtenTableModel btmKlaeger;
	public JTable tabBeklagte;
	public BeteiligtenTableModel btmBeklagte;

	private final AbstractAction neuerKlaegerAction = new StreitgenossenBeteiligterNeuAction(this, KLAEGER,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(31021));
	private final AbstractAction neuerBeklagterAction = new StreitgenossenBeteiligterNeuAction(this, BEKLAGTE,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(32021));
	private final AbstractAction neuerDrittwiderbeklagterAction = new StreitgenossenBeteiligterNeuAction(this,
			DRITTWIDERBEKLAGTE, (LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(33121));

	public JTable tabDrittwiderbeklagte;
	public BeteiligtenTableModel btmDrittwiderbeklagte;
	private JScrollPane scpDwbk;
	private JButton btnWiderklageAnzeigen;
	private JButton btnDrittwiderbeklagterNeu;
	private JButton btnKlDwblGesamtsch;
	private AbstractAction bkGesamtschuldnerAction = new StreitgenossenGesamtschuldnerBkAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(32031));
	private final AbstractAction widerKlageAktivierenAction = new StreitgenossenWiderklageAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(33001));
	private final AbstractAction klDwbkGesamtschAction = new StreitgenossenGesamtschuldnerDwbkAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(33131));
	private final AbstractAction baumbachAction = new StreitgenossenBaumbachAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(34001));

	public final BeschriebeneAktion gesamtschuldnerschaftDwbkExtrahierenAction = new StreitgenossenGesamtschuldDwbkExtractAction(
			this, (LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(39111));
	public final BeschriebeneAktion gesamtschuldnerschaftExtrahierenAction = new StreitgenossenGesamtschuldBkExtractAction(
			this, (LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(39211));
	public AbstractAction xmlLesenAction = new StreitgenossenXJustizLesenAction(this, getKostenrechner(),
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(61031));
	public AbstractAction baumbachAssistentAction = new StreitgenossenAssistentAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(62051));

	private boolean wirtschaftlichIdentisch = false;
	public static final String FEHLER_BERECHNUNG = "Fehler bei der Berechnung! ";

	private static final String FEHLER_GESAMTSCHULD_NIX_LOESCHEN = "Einträge können nicht entfernt werden, wenn bereits eine Gesamtschuldnerschaft vorhanden ist.";
	public static final String FEHLER_GESAMTSCHULDNERSCHAFT = "Fehler beim Hinzufügen einer Gesamtschuldnerschaft";
	public static final String FEHLER_NIX_PARTEIEN = "Fehler: Keine Kläger und/oder Beklagten vorhanden";
	public static final String FEHLER_NIX_PARTEIEN_LANG = "Die Berechnung erfordert einen Kläger und mindestens einen Beklagten.";
	private static final WiderklageTableRendererEditor WIDERKLAGE_RENDERER_EDITOR = new WiderklageTableRendererEditor();
	public static final BeteiligtenTableRenderer BETEILIGTEN_RENDERER = new BeteiligtenTableRenderer();
	private ParsendUndBauend<Streitgenossen> pruefer = new StreitgenossenParser();

	/**
	 * Der Konstruktor ruft nur die Oberklasse auf.
	 * 
	 */
	private StreitgenossenPanel() {
		super();
	}

	/**
	 * @return gibt {@link #instance} als BaumbachPanel zurück.
	 */
	public static StreitgenossenPanel getInstance() {
		return instance;
	}

	/**
	 * Die Methode setzt die Oberflächenelemente für die Registerkarte 2
	 * Streitgenossen
	 * 
	 * 
	 * @see eu.gronos.kostenrechner.view.KostenJPanel#buildPanel()
	 */
	@Override
	public void buildPanel() {
		setWiderKlageAnzeigen(false);

		leereWerte();

		JPanel pnKlaeger = // createPanel
				addNewPanel((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(31000), new GridBagLayout());
		tabKlaeger = (JTable) beschrifter.beschrifte(new JTable(btmKlaeger),
				NameContainerSammlung.BESCHRIFTUNGEN.get(31011));
		setzeBeteiligtenRenderer(tabKlaeger);
//		JScrollPane scpKlaeger = (JScrollPane) 
		beschrifter.beschrifteVergittert(new JScrollPane(tabKlaeger), NameContainerSammlung.BESCHRIFTUNGEN.get(31010),
				pnKlaeger);
		entferneTastaturFalle(tabKlaeger);
		baueEditModeEin(tabKlaeger);

		JPanel pnBeklagte = // createPanel
				addNewPanel((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(32000), new GridBagLayout());
		tabBeklagte = (JTable) beschrifter.beschrifte(new JTable(btmBeklagte),
				NameContainerSammlung.BESCHRIFTUNGEN.get(32011));
		setzeBeteiligtenRenderer(tabBeklagte);
//		JScrollPane scpBeklagte = (JScrollPane) 
		beschrifter.beschrifteVergittert(new JScrollPane(tabBeklagte), NameContainerSammlung.BESCHRIFTUNGEN.get(32010),
				pnBeklagte);
		entferneTastaturFalle(tabBeklagte);
		baueEditModeEin(tabBeklagte);
//		JButton btnKlgerNeu = .toButton()
		beschrifter.add((BeschriebeneAktion) neuerKlaegerAction, pnKlaeger);

		JPanel pnWiderklage = // createPanel
				addNewPanel((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(33000), new GridBagLayout());
		tabDrittwiderbeklagte = (JTable) beschrifter.beschrifte(new JTable(btmDrittwiderbeklagte),
				NameContainerSammlung.BESCHRIFTUNGEN.get(33111));
		setzeBeteiligtenRenderer(tabDrittwiderbeklagte);
		scpDwbk = (JScrollPane) beschrifter.beschrifteVergittert(new JScrollPane(tabDrittwiderbeklagte),
				NameContainerSammlung.BESCHRIFTUNGEN.get(33110), pnWiderklage);
		entferneTastaturFalle(tabDrittwiderbeklagte);
		baueEditModeEin(tabDrittwiderbeklagte);
		//// .toButton()
		btnDrittwiderbeklagterNeu = (JButton) beschrifter.add((BeschriebeneAktion) neuerDrittwiderbeklagterAction,
				pnWiderklage);

//		JButton btnBeklagterNeu = .toButton()
		beschrifter.add((BeschriebeneAktion) neuerBeklagterAction, pnBeklagte);

		// .toButton()
		btnKlDwblGesamtsch = (JButton) beschrifter.add((BeschriebeneAktion) klDwbkGesamtschAction, pnWiderklage);
		// .toButton()
		btnWiderklageAnzeigen = (JButton) beschrifter.add((BeschriebeneAktion) widerKlageAktivierenAction,
				pnWiderklage);

//		JButton btnBkGesamtsch = .toButton()
		beschrifter.add((BeschriebeneAktion) bkGesamtschuldnerAction, pnBeklagte);

//		JButton btnBerechnen = (JButton) // .toButton();
		beschrifter.add((BeschriebeneAktion) baumbachAction, this);

		setWiderKlageAnzeigen(false);
		// zum Panel hinzufügen
		// .add(
//		beschrifter.testUnVergittert(pnKlaeger, scpKlaeger,
//				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 2, GridBagConstraints.BOTH, false));
		// pnKlaeger.add(
//		beschrifter.testUnVergittert(pnKlaeger, btnKlgerNeu,
//				GitterBeutelBeschraenkungen.getInstance(0, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));

//		beschrifter.testUnVergittert(pnWiderklage, // .add(
//				scpDwbk, GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 2, GridBagConstraints.BOTH, false));
		// pnWiderklage.add(
//		beschrifter.testUnVergittert(pnWiderklage, btnDrittwiderbeklagterNeu,
//				GitterBeutelBeschraenkungen.getInstance(0, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));
		// pnWiderklage.add(
//		beschrifter.testUnVergittert(pnWiderklage, btnKlDwblGesamtsch,
//				GitterBeutelBeschraenkungen.getInstance(0, 3, 1, 1, GridBagConstraints.HORIZONTAL, false));
		// pnWiderklage.add(
//		beschrifter.testUnVergittert(pnWiderklage, btnWiderklageAnzeigen,
//				GitterBeutelBeschraenkungen.getInstance(0, 4, 1, 1, GridBagConstraints.NONE, false));

//		beschrifter.testUnVergittert(pnBeklagte, // .add(
//				scpBeklagte, GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 2, GridBagConstraints.BOTH, false));
		// pnBeklagte.add(
//		beschrifter.testUnVergittert(pnBeklagte, btnBeklagterNeu,
//				GitterBeutelBeschraenkungen.getInstance(0, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));
		// pnBeklagte.add(
//		beschrifter.testUnVergittert(pnBeklagte, btnBkGesamtsch,
//				GitterBeutelBeschraenkungen.getInstance(0, 3, 1, 1, GridBagConstraints.HORIZONTAL, false));

		// add(
//		beschrifter.testUnVergittert(this, pnKlaeger,
//				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.BOTH, false));
//		add(
//		beschrifter.testUnVergittert(this, pnWiderklage,
//				GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1, GridBagConstraints.BOTH, false));
		// add(
//		beschrifter.testUnVergittert(this,pnBeklagte, GitterBeutelBeschraenkungen.getInstance(1, 0, 1, 2, GridBagConstraints.BOTH, false));
		// add(
//		beschrifter.testUnVergittert(this, btnBerechnen,
//				GitterBeutelBeschraenkungen.getInstance(0, 2, 2, 1, GridBagConstraints.HORIZONTAL, false));

		PopupMenuBuilder builder = new PopupMenuBuilder();

		builder.build((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(39100), tabKlaeger, new Integer[] { 1 },
				(BeschriebeneAktion) neuerKlaegerAction, (BeschriebeneAktion) klDwbkGesamtschAction,
				(BeschriebeneAktion) gesamtschuldnerschaftDwbkExtrahierenAction,
				(BeschriebeneAktion) eintragEntfernenAktion);
		builder.build((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(39200), tabBeklagte, new Integer[] { 1 },
				(BeschriebeneAktion) neuerBeklagterAction, (BeschriebeneAktion) bkGesamtschuldnerAction,
				(BeschriebeneAktion) gesamtschuldnerschaftExtrahierenAction,
				(BeschriebeneAktion) eintragEntfernenAktion);
		builder.build((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(39300), tabDrittwiderbeklagte,
				new Integer[] { 1 }, (BeschriebeneAktion) neuerDrittwiderbeklagterAction,
				(BeschriebeneAktion) klDwbkGesamtschAction,
				(BeschriebeneAktion) gesamtschuldnerschaftDwbkExtrahierenAction,
				(BeschriebeneAktion) eintragEntfernenAktion);

		logDauer();
	}

	/**
	 * Die Methode leert die Baumbach-Tabellen und füllt sie mit den Daten aus dem
	 * Untercontainer {@link eu.gronos.kostenrechner.data.baumbach.Streitgenossen}
	 * 
	 * @param container ein Untercontainer
	 *                  {@link eu.gronos.kostenrechner.data.baumbach.Streitgenossen}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.StreitgenossenTabellenBehandelnd#setzeWerte(eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer.Streitgenossen)
	 */
	@Override
	public void setzeWerte(final Streitgenossen container) {
		setWiderKlageAnzeigen(container.widerklage);
		setWirtschaftlichIdentisch(container.wirtschaftlichIdentisch);
		btmKlaeger = new BeteiligtenTableModel(KLAEGER, isWiderKlageAnzeigen(), container.klaeger);
		tabKlaeger.setModel(btmKlaeger);
		btmDrittwiderbeklagte = new BeteiligtenTableModel(DRITTWIDERBEKLAGTE, isWiderKlageAnzeigen(),
				container.drittwiderbeklagte);
		tabDrittwiderbeklagte.setModel(btmDrittwiderbeklagte);
		btmBeklagte = new BeteiligtenTableModel(BEKLAGTE, isWiderKlageAnzeigen(), container.beklagte);
		tabBeklagte.setModel(btmBeklagte);
		Kostenrechner.getLogger().info("Fertig!");
	}

	/**
	 * Die Methode dient dazu, alle Programmzustände in einen
	 * {@link VerfahrensDatenContainer} zu packen
	 * 
	 * @return den Untercontainer von
	 *         {@link VerfahrensDatenContainer#streitgenossen}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.WerteLesbar#baueRueckgabewert()
	 */
	@Override
	public Streitgenossen baueRueckgabewert() {
		((StreitgenossenParser) pruefer)
				.setzeSonstigeWerte(new Widerklage(isWiderKlageAnzeigen(), isWirtschaftlichIdentisch()))
				.leseComponents(tabKlaeger, tabBeklagte, tabDrittwiderbeklagte).parseEingabe();
		return pruefer.getRueckgabewert();
	}

	/**
	 * Die Methode setWiderklageAnzeigen dient zum Aktivieren der Widerklagen. Bei
	 * allen TableModels widerKlageAnzeigen setzen, Dwbk-Tabelle und Schaltflächen
	 * wieder auf Visible setzen, eigene Schaltflächen ausblenden.
	 * 
	 * @param widerKlageAnzeigen der Wert für widerKlageAnzeigen, der gesetzt werden
	 *                           soll.
	 */
	public void setWiderKlageAnzeigen(boolean widerKlageAnzeigen) {
		this.widerKlageAnzeigen = widerKlageAnzeigen;
		if (scpDwbk != null)
			scpDwbk.setVisible(widerKlageAnzeigen);
		if (btnDrittwiderbeklagterNeu != null)
			btnDrittwiderbeklagterNeu.setVisible(widerKlageAnzeigen);
		if (btnKlDwblGesamtsch != null)
			btnKlDwblGesamtsch.setVisible(widerKlageAnzeigen);
		if (btnWiderklageAnzeigen != null)
			btnWiderklageAnzeigen.setVisible(!widerKlageAnzeigen);
	}

	/**
	 * Die Methode leert die Tabellen der BaumbachBeteiligten
	 * 
	 * 
	 * @see eu.gronos.kostenrechner.view.KostenJPanel#leereWerte()
	 */
	@Override
	public void leereWerte() {
		btmKlaeger = new BeteiligtenTableModel(KLAEGER, isWiderKlageAnzeigen(), new BaumbachBeteiligtenListe());
		btmDrittwiderbeklagte = new BeteiligtenTableModel(DRITTWIDERBEKLAGTE, isWiderKlageAnzeigen());
		btmBeklagte = new BeteiligtenTableModel(BEKLAGTE, isWiderKlageAnzeigen(), new BaumbachBeteiligtenListe());
		if (tabKlaeger != null)
			tabKlaeger.setModel(btmKlaeger);
		if (tabBeklagte != null)
			tabBeklagte.setModel(btmBeklagte);
		if (tabDrittwiderbeklagte != null)
			tabDrittwiderbeklagte.setModel(btmDrittwiderbeklagte);
	}

	/**
	 * Die Methode setzt die DefaultRenderer für Beteiligten-JTables
	 * (BeteiligtenRenderer und EuroRenderer) und bestimmt, dass sie den Viewport
	 * ausfüllen müssen.
	 * 
	 * @param table JTable für Beteiligtendaten
	 * 
	 */
	private void setzeBeteiligtenRenderer(JTable table) {
		table.setDefaultRenderer(Beteiligter.class, BETEILIGTEN_RENDERER);
		table.setDefaultEditor(Beteiligter.class,
				new BeteiligtenCellEditor(((BeteiligtenTableModel) table.getModel()).getBeteiligtenTyp()));

		table.setDefaultRenderer(Double.class, EURO_RENDERER);
		table.setDefaultRenderer(Euro.class, EURO_RENDERER);
		table.setDefaultEditor(Double.class, EURO_EDITOR);
		table.setDefaultEditor(Euro.class, EURO_EDITOR);
		table.setDefaultRenderer(Boolean.class, WIDERKLAGE_RENDERER_EDITOR);
		table.setDefaultEditor(Boolean.class, WIDERKLAGE_RENDERER_EDITOR);
		table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		table.getTableHeader().setDefaultRenderer(HEADER_RENDERER);
	}

	/**
	 * Die Methode prüft, ob die Tabellen für die BaumbachBeteiligten auf Kläger-
	 * und auf Beklagtenseite Werte enthalten.
	 * 
	 * @return true, wenn btmKlaeger und btmBeklagte Beteiligte enthalten; false,
	 *         wenn auch nur eine der beiden leer ist.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TabellenWertePruefend#pruefeTabellen()
	 */
	@Override
	public boolean pruefeTabellen() {
		return btmKlaeger.getRowCount() > 0 && btmBeklagte.getRowCount() > 0;
	}

	/**
	 * Die Methode dient dazu, eine Referenz auf das Hauptfenster zu übermitteln
	 * 
	 * @param kostenrechner
	 * 
	 * @see eu.gronos.kostenrechner.view.KostenJPanel#setKostenrechner(eu.gronos.kostenrechner.Kostenrechner)
	 */
	@Override
	public void setKostenrechner(Kostenrechner kostenrechner) {
		this.kostenrechner = kostenrechner;
	}

	/**
	 * Entfernt eine Zeile aus einer Beteiligten-Tabelle. Es wird zuvor geprüft, ob
	 * die entsprechende Tabelle bereits Gesamtschuldnerschaften enthält; dann ist
	 * ein Entfernen aus Gründen möglicher Verweise nicht möglicht. Es sei denn: die
	 * Gesamtschuldnerschaft selbst wird entfernt.
	 * 
	 * @param table die JTable, aus der die Zeile gelöscht werden soll
	 * @param model das BeteiligtenTableModel, aus der die Zeile gelöscht werden
	 *              soll.
	 * 
	 */
	public void entferneBeteiligtenEintrag(JTable table, BeteiligtenTableModel model) {
		if (model == null)
			return;
		if (model.getAllValues() != null && model.getAllValues().enthaeltGesamtschuldner()) {
			if (table.getSelectedRow() > -1 && !model.getRow(table.getSelectedRow()).isGesamtschuldnerschaft()) {
				FehlerHelper.zeigeFehler(FEHLER_GESAMTSCHULD_NIX_LOESCHEN,
						new IllegalArgumentException(FEHLER_GESAMTSCHULD_NIX_LOESCHEN));
				return;
			}
		} else if (model.getBeteiligtenTyp() == KLAEGER && (btmDrittwiderbeklagte.getAllValues() != null
				&& btmDrittwiderbeklagte.getAllValues().enthaeltGesamtschuldner())) {
			if (table.getSelectedRow() > -1 && !model.getRow(table.getSelectedRow()).isGesamtschuldnerschaft()) {
				FehlerHelper.zeigeFehler(FEHLER_GESAMTSCHULD_NIX_LOESCHEN,
						new IllegalArgumentException(FEHLER_GESAMTSCHULD_NIX_LOESCHEN));
				return;
			}
		}
		((EintragEntfernenAktion) eintragEntfernenAktion).entferneTabellenEintrag(table, model);
	}

	/**
	 * @return gibt {@link #kostenrechner} als Kostenrechner zurück.
	 */
	public Kostenrechner getKostenrechner() {
		return kostenrechner;
	}

	/**
	 * @return gibt {@link #wirtschaftlichIdentisch} als boolean zurück.
	 */
	public boolean isWirtschaftlichIdentisch() {
		return wirtschaftlichIdentisch;
	}

	/**
	 * @param wirtschaftlichIdentisch d. {@link #wirtschaftlichIdentisch}, d.
	 *                                gesetzt werden soll als boolean
	 */
	public void setWirtschaftlichIdentisch(boolean wirtschaftlichIdentisch) {
		this.wirtschaftlichIdentisch = wirtschaftlichIdentisch;
	}

	/**
	 * @return gibt {@link #widerKlageAnzeigen} als boolean zurück.
	 */
	public boolean isWiderKlageAnzeigen() {
		return widerKlageAnzeigen;
	}

	/**
	 * Die Methode gibt das zu einem {@link Beteiligter#beteiligtenTypAsInt()}
	 * zugehörige {@link BeteiligtenTableModel} zurück.
	 * 
	 * @param typ {@link Beteiligter#KLAEGER}, {@link Beteiligter#BEKLAGTE} oder
	 *            {@link Beteiligter#DRITTWIDERBEKLAGTE}
	 * @return das dazu gehörige {@link BeteiligtenTableModel}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TableModelGebend#getTableModel(int)
	 */
	@Override // int
	public BeteiligtenTableModel getTableModel(int index) {
		final BeteiligtenTyp typ = (index < 0 || index >= BeteiligtenTyp.values().length) //
				? null
				: BeteiligtenTyp.of(index);
		switch (typ) {
		case KLAEGER:
			return btmKlaeger;
		case DRITTWIDERBEKLAGTE:
			return btmDrittwiderbeklagte;
		case BEKLAGTE:
			return btmBeklagte;
		default:
			return null;
		}
	}

	/**
	 * Die Methode dient dazu, dass eine {@link BeschriebeneAktion} den Focus auf
	 * die {@link JComponent} legen kann, die sie aufruft.
	 * 
	 * @return <code>true</code>, wenn der Aufruf wahrscheinlich Erfolg hat;
	 *         <code>false</code>, wenn er zum Scheitern verurteilt ist.
	 * 
	 * @see javax.swing.JComponent#requestFocusInWindow()
	 */
	@Override
	public boolean requestFocusInWindow() {
		getKostenrechner().getTabbedPane().setSelectedIndex(Streitgenossen.TAB_ID);
		return super.requestFocusInWindow();
	}

	@Override
	public BeschriebeneAktion getExtrahierenAktion(Beteiligter.BeteiligtenTyp beteiligtenTyp)
			throws IllegalArgumentException {
		if (BEKLAGTE == beteiligtenTyp)
			return gesamtschuldnerschaftExtrahierenAction;
		else if (DRITTWIDERBEKLAGTE == beteiligtenTyp)
			return gesamtschuldnerschaftDwbkExtrahierenAction;
		else
			throw new IllegalArgumentException(String.format("Unbekannter Beteiligtentyp: %d", beteiligtenTyp));
	}

}
