/**
 * WiderklageTableRendererEditor.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.view.baumbach;

import java.awt.Component;
import java.awt.event.MouseEvent;

import javax.swing.AbstractCellEditor;
import javax.swing.JCheckBox;
import javax.swing.JRadioButton;
import javax.swing.JTable;
import javax.swing.JToggleButton;
import javax.swing.SwingConstants;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;

import eu.gronos.kostenrechner.controller.AllzweckHorcher;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp;
import eu.gronos.kostenrechner.model.baumbach.BeteiligtenTableModel;

/**
 * Ein TableCellRenderer und TableCellEditor, der über eine JCheckBox oder
 * JRadioButton anzeigt, ob der Beteiligte Widerkläger bzw. Widerbeklagter ist.
 * 
 * @author Peter Felix Schuster (setrok)
 * @date 07.08.2014
 */
public class WiderklageTableRendererEditor extends AbstractCellEditor implements TableCellRenderer, TableCellEditor {

	private static final long serialVersionUID = 1449507097445915897L;
	private JToggleButton editorToggle = null;
	private final AllzweckHorcher EDITOR_STOPPER = new AllzweckHorcher() {
		@Override
		public void mouseEntered(MouseEvent me) {
			stopCellEditing();
		}

		@Override
		public void mouseExited(MouseEvent me) {
			stopCellEditing();
		}

		@Override
		public void mouseMoved(MouseEvent me) {
			stopCellEditing();
		}
	};

	/**
	 * Konstruktor:
	 * 
	 */
	public WiderklageTableRendererEditor() {
		super();
	}

	/**
	 * Returns the value contained in the editor.
	 * 
	 * @return the value contained in the editor
	 * 
	 * @see javax.swing.CellEditor#getCellEditorValue()
	 */
	@Override
	public Object getCellEditorValue() {
		boolean b = editorToggle == null ? false : editorToggle.isSelected();
		return new Boolean(b);
	}

	/**
	 * Sets an initial value for the editor. This will cause the editor to
	 * stopEditing and lose any partially edited value if the editor is editing when
	 * this method is called. Returns the component that should be added to the
	 * client's Component hierarchy. Once installed in the client's hierarchy this
	 * component will then be able to draw and receive user input.
	 * 
	 * 
	 * @param table      the JTable that is asking the renderer to draw; can be null
	 * @param value      the value of the cell to be rendered. It is up to the
	 *                   specific renderer to interpret and draw the value. For
	 *                   example, if value is the string "true", it could be
	 *                   rendered as a string or it could be rendered as a check box
	 *                   that is checked. null is a valid value
	 * @param isSelected true if the cell is to be rendered with the selection
	 *                   highlighted; otherwise false
	 * @param row        the row index of the cell being drawn. When drawing the
	 *                   header, the value of row is -1
	 * @param column     the column index of the cell being drawn
	 * @return einen JToggleButton und zwar einen JRadioButton bei Beklagten, sonst
	 *         eine JCheckBox
	 * 
	 * @see javax.swing.table.TableCellEditor#getTableCellEditorComponent(javax.swing.JTable,
	 *      java.lang.Object, boolean, int, int)
	 */
	@Override
	public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) {
		editorToggle = (JToggleButton) getTableCellRendererComponent(table, value, true, true, row, column);
		editorToggle.addMouseMotionListener(EDITOR_STOPPER);
		return editorToggle;
	}

	/**
	 * Returns the component used for drawing the cell. This method is used to
	 * configure the renderer appropriately before drawing. The TableCellRenderer is
	 * also responsible for rendering the the cell representing the table's current
	 * DnD drop location if it has one. If this renderer cares about rendering the
	 * DnD drop location, it should query the table directly to see if the given row
	 * and column represent the drop location:
	 * 
	 * During a printing operation, this method will be called with isSelected and
	 * hasFocus values of false to prevent selection and focus from appearing in the
	 * printed output. To do other customization based on whether or not the table
	 * is being printed, check the return value from
	 * JComponent.isPaintingForPrint().
	 * 
	 * @param table      the JTable that is asking the renderer to draw; can be null
	 * @param value      the value of the cell to be rendered. It is up to the
	 *                   specific renderer to interpret and draw the value. For
	 *                   example, if value is the string "true", it could be
	 *                   rendered as a string or it could be rendered as a check box
	 *                   that is checked. null is a valid value
	 * @param isSelected true if the cell is to be rendered with the selection
	 *                   highlighted; otherwise false
	 * @param hasFocus   if true, render cell appropriately. For example, put a
	 *                   special border on the cell, if the cell can be edited,
	 *                   render in the color used to indicate editing
	 * @param row        the row index of the cell being drawn. When drawing the
	 *                   header, the value of row is -1
	 * @param column     the column index of the cell being drawn
	 * @return einen JToggleButton und zwar einen JRadioButton bei Beklagten, sonst
	 *         eine JCheckBox
	 * 
	 * @see javax.swing.table.TableCellRenderer#getTableCellRendererComponent(javax.swing.JTable,
	 *      java.lang.Object, boolean, boolean, int, int)
	 */
	@Override
	public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus,
			int row, int column) {
		if (!(value instanceof Boolean))
			return null;
		boolean b = ((Boolean) value).booleanValue();
		JToggleButton toggle = null;
		BeteiligtenTableModel btm = null;
		if (table.getModel() instanceof BeteiligtenTableModel) {
			btm = (BeteiligtenTableModel) table.getModel();
			if (btm.getBeteiligtenTyp() == BeteiligtenTyp.BEKLAGTE)
				toggle = new JRadioButton();
			else
				toggle = new JCheckBox();
		} else
			return null;
		toggle.setSelected(b);
		toggle.setText(null);
		toggle.setHorizontalAlignment(SwingConstants.CENTER);
		toggle.setOpaque(true);
		if (isSelected) {
			toggle.setBackground(table.getSelectionBackground());
			toggle.setForeground(table.getSelectionForeground());
		} else {
			toggle.setBackground(table.getBackground());
			toggle.setForeground(table.getForeground());
		}
		/*
		 * Unter Mac OS X werden JRadioButtons abgeschnitten. Mal sehen, ob das was
		 * hilft.
		 */
		int height = toggle.getMinimumSize().height;
		height -= (table.getIntercellSpacing().height * 2);
		if (table.getRowHeight(row) < height)
			table.setRowHeight(row, height);
		// Die Breite etwas herabsetzen...
		int width = toggle.getMinimumSize().width;
		if (width < table.getTableHeader().getColumnModel().getColumn(column).getMinWidth())
			width = table.getTableHeader().getColumnModel().getColumn(column).getMinWidth();
		table.getColumnModel().getColumn(column).setPreferredWidth(width);
		return toggle;
	}
}
